const express = require('express');
const axios = require('axios');
const cors = require('cors');
const bodyParser = require('body-parser');
const fs = require('fs');
const path = require('path');
const { v4: uuidv4 } = require('uuid');
const multer = require('multer');
require('dotenv').config();
const rateLimit = require('express-rate-limit');

// Multer setup for file uploads
const upload = multer({
    storage: multer.memoryStorage(),
    limits: { fileSize: 50 * 1024 * 1024 } // 50MB max
});

// Design Repository Config
const DESIGNS_DIR = path.join(__dirname, 'data', 'designs');
if (!fs.existsSync(DESIGNS_DIR)) {
    fs.mkdirSync(DESIGNS_DIR, { recursive: true });
}

// Multer storage for designs
const designStorage = multer.diskStorage({
    destination: (req, file, cb) => cb(null, DESIGNS_DIR),
    filename: (req, file, cb) => cb(null, file.originalname.replace(/\s+/g, '_'))
});
const uploadDesign = multer({ storage: designStorage });

const app = express();
const PORT = process.env.PORT || 3000;

app.use(cors());
app.use(bodyParser.json());
app.use(express.static('public'));

// DEBUG LOGGER
app.use((req, res, next) => {
    if (req.path.startsWith('/api/')) {
        console.log(`[REQUEST] ${req.method} ${req.path} | Auth: ${!!req.headers.authorization}`);
    }
    next();
});

// In-memory session store (simple enough for single instance)
const SESSIONS = new Set();
const VALID_TOKEN_AGE = 24 * 60 * 60 * 1000; // 24 hours

// Login Rate Limiter
const loginLimiter = rateLimit({
    windowMs: 15 * 60 * 1000, // 15 minutes
    max: 5, // Limit each IP to 5 requests per windowMs
    message: { error: 'Çok fazla giriş denemesi yaptınız. Lütfen 15 dakika sonra tekrar deneyin.' },
    standardHeaders: true,
    legacyHeaders: false,
});

// Login Endpoint
app.post('/api/login', loginLimiter, (req, res) => {
    const { username, password } = req.body;
    const adminUser = process.env.ADMIN_USER || 'admin';
    const adminPass = process.env.ADMIN_PASS || '123456';

    if (username === adminUser && password === adminPass) {
        const token = uuidv4();
        SESSIONS.add(token);
        res.json({ success: true, token });
    } else {
        res.status(401).json({ error: 'Geçersiz kullanıcı adı veya şifre' });
    }
});

// Token Auth Middleware
const tokenAuth = (req, res, next) => {
    // Public paths
    const publicPaths = ['/login.html', '/api/login', '/style.css', '/favicon.ico', '/titlecheck-debug.html'];
    if (publicPaths.includes(req.path) || req.path.startsWith('/node_modules/')) {
        return next();
    }

    // Check Authorization Header for API calls
    const authHeader = req.headers.authorization;
    const token = authHeader && authHeader.split(' ')[1];

    // Allow DEBUG_MODE for TitleCheck debug page
    if (token === 'DEBUG_MODE' && req.path.startsWith('/api/titlecheck/')) {
        return next();
    }

    // Check valid session token
    if (authHeader && authHeader.startsWith('Bearer ') && SESSIONS.has(token)) {
        return next();
    }

    // Handle AJAX/API requests vs Browser navigation
    // FIXED: Always return JSON for /api/ routes, never redirect
    const accept = req.headers.accept || '';
    // console.log(`[AuthDebug] Path: ${req.path}, Accept: ${accept}, Auth: ${!!authHeader}`); 

    if (req.path.startsWith('/api/') || req.xhr || accept.indexOf('json') > -1) {
        return res.status(401).json({ error: 'Authentication required' });
    } else {
        // Redirect browser to login
        res.redirect('/login.html');
    }
};

// Protect all routes
app.use(tokenAuth);

// Simple JSON DB Helper
const DB_FILE = path.join(__dirname, 'data', 'db.json');

const getDb = () => {
    if (!fs.existsSync(DB_FILE)) return [];
    try {
        const content = fs.readFileSync(DB_FILE, 'utf8');
        return content ? JSON.parse(content) : [];
    } catch (e) {
        console.error('Database read error:', e);
        return [];
    }
};

const saveDb = (data) => {
    fs.writeFileSync(DB_FILE, JSON.stringify(data, null, 2));
};

// Cloudflare API Helper
const cfApi = async (method, endpoint, data = null, token, email = null) => {
    try {
        const headers = { 'Content-Type': 'application/json' };

        if (email && token) {
            headers['X-Auth-Email'] = email;
            headers['X-Auth-Key'] = token;
        } else {
            headers['Authorization'] = `Bearer ${token}`;
        }

        const config = {
            method,
            url: `https://api.cloudflare.com/client/v4${endpoint}`,
            headers,
            data
        };
        const response = await axios(config);
        return response.data;
    } catch (error) {
        console.error(`API Error [${endpoint}]:`, error.response?.data || error.message);
        throw error.response?.data || error;
    }
};

// ==================== WORDPRESS MANAGER CONFIG ====================
const WP_SITES_FILE = path.join(__dirname, 'data', 'wp_sites.json');

const getWpSites = () => {
    if (!fs.existsSync(WP_SITES_FILE)) return [];
    try {
        const content = fs.readFileSync(WP_SITES_FILE, 'utf8');
        return content ? JSON.parse(content) : [];
    } catch (e) {
        return [];
    }
};

const saveWpSites = (data) => {
    fs.writeFileSync(WP_SITES_FILE, JSON.stringify(data, null, 2));
};

// ==================== WORDPRESS MANAGER ROUTES ====================

// 1. Get All WP Sites
app.get('/api/wp/sites', (req, res) => {
    const sites = getWpSites();
    // Return sites without sensitive keys for list, or with keys if needed by frontend proxy
    //Ideally we shouldn't expose keys, but for this single-user tool it's acceptable for simplicity
    res.json(sites);
});

// 2. Add WP Site
app.post('/api/wp/sites', (req, res) => {
    const { name, url, key } = req.body;
    if (!name || !url || !key) {
        return res.status(400).json({ error: "Site ismi, URL ve Erişim Anahtarı gereklidir." });
    }

    const sites = getWpSites();
    const newSite = {
        id: uuidv4(),
        name,
        url: url.replace(/\/$/, ''), // Remove trailing slash
        key,
        createdAt: new Date().toISOString()
    };
    sites.push(newSite);
    saveWpSites(sites);

    res.json(newSite);
});

// 3. Delete WP Site
app.delete('/api/wp/sites/:id', (req, res) => {
    const sites = getWpSites();
    const filtered = sites.filter(s => s.id !== req.params.id);
    saveWpSites(filtered);
    res.json({ success: true });
});

// 4. Proxy Request to WP Agent
app.post('/api/wp/proxy', async (req, res) => {
    const { siteId, action, params } = req.body;

    const sites = getWpSites();
    const site = sites.find(s => s.id === siteId);

    if (!site) {
        return res.status(404).json({ error: "Site bulunamadı." });
    }

    try {
        // console.log(`[WP Proxy] ${action} -> ${site.url}`);

        const response = await axios.post(`${site.url}/wp-helper.php`, {
            action,
            params
        }, {
            headers: {
                'X-API-KEY': site.key, // PHP tarafında HTTP_X_API_KEY olarak görünür
                'Authorization': site.key, // Yedek olarak kalsın
                'Content-Type': 'application/json',
                'User-Agent': 'Cloudflare-Dashboard-Agent'
            },
            timeout: 60000 // 60s timeout for long ops like backups
        });

        res.json(response.data);
    } catch (error) {
        const errorMsg = error.response?.data?.message || error.message;
        console.error(`[WP Proxy Error] ${site.url}:`, errorMsg);

        if (error.response) {
            res.status(error.response.status).json(error.response.data);
        } else {
            res.status(500).json({ status: 'error', message: "Bağlantı hatası: " + errorMsg });
        }
    }
});

// ==================== CLICKAINE CONFIGURATION & HELPER ====================
const CLICKAINE_API_KEY = process.env.CLICKAINE_API_KEY;
const CLICKAINE_API_BASE = process.env.CLICKAINE_API_BASE || 'https://api.clickaine.com';
const TEMPLATE_CAMPAIGN_ID = 760598;
const CAMPAIGN_TEMPLATE = {
    adFormat: 'popunder',
    payType: 'cpm',
    bid: 0.25,
    countries: [298795],
    themeID: 120,
    dailyBudget: 0,
    overallBudget: 12,
    hourlyTrafficLimit: 12000
};

const SUPERHERO_NAMES = [
    'SUPERMAN', 'BATMAN', 'SPIDERMAN', 'IRONMAN', 'THOR', 'HULK', 'WOLVERINE',
    'DEADPOOL', 'AQUAMAN', 'FLASH', 'WONDER', 'CAPTAIN', 'PANTHER', 'STRANGE',
    'VISION', 'HAWK', 'FALCON', 'ANTMAN', 'WASP', 'GROOT', 'ROCKET', 'DRAX',
    'GAMORA', 'NEBULA', 'LOKI', 'THANOS', 'ULTRON', 'JARVIS', 'FURY', 'WIDOW'
];

function generateCampaignName() {
    const hero = SUPERHERO_NAMES[Math.floor(Math.random() * SUPERHERO_NAMES.length)];
    const number = Math.floor(10000 + Math.random() * 90000);
    return `${hero} ${number}`;
}

async function clickaineApiRequest(method, endpoint, data = null) {
    try {
        const response = await axios({
            method,
            url: `${CLICKAINE_API_BASE}${endpoint}`,
            headers: {
                'X-Clickaine-API-Key': CLICKAINE_API_KEY,
                'Content-Type': 'application/json'
            },
            data
        });
        return { success: true, data: response.data };
    } catch (error) {
        return { success: false, error: error.response?.data || error.message };
    }
}

// Helper to generate a Scoped Token using Global Key
async function generateScopedToken(globalKey, email) {
    if (!globalKey || !email) throw new Error("Global Key and Email required for token generation.");

    const zoneGroups = [
        { id: 'e6d2666161e84845a636613608cee8d5' }, // Zone Write
        { id: '4755a26eedb94da69e1066d98aa820be' }, // DNS Write
        { id: 'c03055bc037c4ea9afb9a9f104b7b721' }, // SSL Write
        { id: 'ed07f6c337da4195b4e72a1fb2c6bcae' }, // Page Rules Write
        { id: '3030687196b94b638145a3953da2b699' }, // Zone Settings Write
        { id: 'fb6778dc191143babbfaa57993f1d275' }, // Zone WAF Write
        { id: '9ff81cbbe65c400b97d92c3c1033cab6' }, // Cache Settings Write
        { id: '3b94c49258ec4573b06d51d99b6416c0' }  // Bot Management Write
    ];

    // Clickaine helpers removed from here


    const accountGroups = [
        { id: 'c1fde68c7bcc44588cbb6ddbc16d6480' }, // Account Settings Read
        { id: 'eb56a6953c034b9d97dd838155666f06' }  // Account API Tokens Read
    ];

    try {
        const createResp = await axios.post('https://api.cloudflare.com/client/v4/user/tokens', {
            name: `AutoManager_PRO_${Date.now()}`,
            policies: [
                {
                    effect: 'allow',
                    resources: { 'com.cloudflare.api.account.zone.*': '*' },
                    permission_groups: zoneGroups
                },
                {
                    effect: 'allow',
                    resources: { 'com.cloudflare.api.account.*': '*' },
                    permission_groups: accountGroups
                }
            ]
        }, {
            headers: {
                'X-Auth-Email': email,
                'X-Auth-Key': globalKey,
                'Content-Type': 'application/json'
            }
        });

        if (createResp.data.success) {
            return createResp.data.result.value;
        } else {
            throw new Error(createResp.data.errors?.[0]?.message || 'Failed to generate token');
        }
    } catch (e) {
        throw new Error(`Token Gen Failed: ${e.response?.data?.errors?.[0]?.message || e.message}`);
    }
}

// ==================== API ROUTES ====================

// Get all accounts
app.get('/api/accounts', (req, res) => {
    // console.log('GET /api/accounts called');
    const accounts = getDb();
    res.json(accounts);
});

// Add account
app.post('/api/accounts', async (req, res) => {
    // console.log('POST /api/accounts called');
    const { name, token, email } = req.body;
    if (!name || !token) return res.status(400).json({ error: 'Name and Token required' });

    let finalToken = token;

    try {
        if (email) {
            // Global Key -> Generate Scoped Token
            await cfApi('GET', '/user', null, token, email);
            finalToken = await generateScopedToken(token, email);
        } else {
            // Direct Token
            await cfApi('GET', '/user/tokens/verify', null, token);
        }
    } catch (e) {
        const msg = e.errors?.[0]?.message || e.message || 'Invalid Credentials';
        return res.status(400).json({ error: `Cloudflare Auth Failed: ${msg}` });
    }

    const accounts = getDb();
    const newAccount = { id: uuidv4(), name, token: finalToken, createdAt: new Date().toISOString() };
    accounts.push(newAccount);
    saveDb(accounts);

    res.json(newAccount);
});

// Delete account
app.delete('/api/accounts/:id', (req, res) => {
    const accounts = getDb();
    const filtered = accounts.filter(a => a.id !== req.params.id);
    saveDb(filtered);
    res.json({ success: true });
});

// Get Zones for an Account
app.get('/api/accounts/:id/zones', async (req, res) => {
    const accounts = getDb();
    const account = accounts.find(a => a.id === req.params.id);

    if (!account) return res.status(404).json({ error: 'Account not found' });

    try {
        const data = await cfApi('GET', '/zones?per_page=50', null, account.token);
        const zones = data.result.map(z => ({
            id: z.id,
            name: z.name,
            status: z.status,
            name_servers: z.name_servers
        }));
        res.json(zones);
    } catch (e) {
        res.status(500).json({ error: e.message });
    }
});

// Delete Zone
app.delete('/api/zones/:id', async (req, res) => {
    const token = req.headers.authorization?.split(' ')[1]; // Get session token
    // In a real app we'd map session token to CF token, but here we likely need the CF Account Token passed or looked up
    // For simplicity, we'll look up the account from the DB that owns this zone (expensive) or expect accountId in query/header
    // BETTER STRATEGY: Receive accountId in query params to find the correct CF Token

    const accountId = req.query.accountId;
    if (!accountId) return res.status(400).json({ error: 'Account ID required' });

    const accounts = getDb();
    const account = accounts.find(a => a.id === accountId);
    if (!account) return res.status(404).json({ error: 'Account not found' });

    try {
        await cfApi('DELETE', `/zones/${req.params.id}`, null, account.token);
        res.json({ success: true, message: 'Zone deleted' });
    } catch (e) {
        res.status(500).json({ error: e.message });
    }
});

// Process Site (Automation)
app.post('/api/process-site', async (req, res) => {
    let { token, domain, ip, subdomains, config } = req.body;
    const logs = [];
    const log = (msg) => logs.push(`[${new Date().toLocaleTimeString()}] ${msg}`);

    if (domain) {
        domain = domain.replace(/^https?:\/\//, '').replace(/\/$/, '').toLowerCase();
    }

    try {
        log(`Starting process for ${domain}`);

        // Verify Token
        try {
            await cfApi('GET', '/user/tokens/verify', null, token);
            log('API Token verified successfully.');
        } catch (e) {
            throw new Error('Invalid API Token.');
        }

        // Get Account ID
        const getAccountId = async (token) => {
            const data = await cfApi('GET', '/accounts', null, token);
            if (data.result && data.result.length > 0) return data.result[0].id;
            throw new Error('No Account ID found.');
        };

        // Add Zone
        let zoneId = null;
        try {
            log('Attempting to create zone...');
            const zoneData = await cfApi('POST', '/zones', {
                name: domain,
                account: { id: (await getAccountId(token)) },
                jump_start: true
            }, token);
            zoneId = zoneData.result.id;
            log(`Zone created. ID: ${zoneId}`);
        } catch (e) {
            if (e.errors?.[0]?.message?.includes('already exists')) {
                log('Zone already exists, fetching ID...');
                const zones = await cfApi('GET', `/zones?name=${domain}`, null, token);
                if (zones.result?.length > 0) {
                    zoneId = zones.result[0].id;
                    log(`Found existing zone ID: ${zoneId}`);
                } else {
                    throw new Error('Could not find existing zone ID.');
                }
            } else {
                throw e;
            }
        }

        if (!zoneId) throw new Error('Failed to obtain Zone ID.');

        // DNS Records
        log('Configuring DNS records...');
        const addDns = async (type, name, content) => {
            try {
                const records = await cfApi('GET', `/zones/${zoneId}/dns_records?type=${type}&name=${name}`, null, token);
                if (records.result?.length > 0) {
                    await cfApi('PUT', `/zones/${zoneId}/dns_records/${records.result[0].id}`, { type, name, content, proxied: true }, token);
                    log(`Updated DNS: ${type} ${name}`);
                } else {
                    await cfApi('POST', `/zones/${zoneId}/dns_records`, { type, name, content, proxied: true }, token);
                    log(`Created DNS: ${type} ${name}`);
                }
            } catch (e) {
                log(`DNS Error ${name}: ${e.errors?.[0]?.message || e.message}`);
            }
        };

        await addDns('A', domain, ip);
        await addDns('CNAME', `www.${domain}`, domain);

        if (subdomains) {
            const subs = subdomains.split(/[\s,]+/).filter(s => s.trim());
            for (const sub of subs) {
                await addDns('CNAME', `${sub}.${domain}`, domain);
            }
        }

        // Settings
        log('Applying settings...');
        const settings = ['always_use_https', 'ssl', 'early_hints', 'websockets'];
        for (const s of settings) {
            try {
                await cfApi('PATCH', `/zones/${zoneId}/settings/${s}`, { value: s === 'ssl' ? 'full' : 'on' }, token);
                log(`Setting ${s} updated.`);
            } catch (e) {
                log(`Warning: ${s} failed.`);
            }
        }

        // WAF (Geo Blocking)
        if (config?.geoBlock) {
            log('Configuring Geo Blocking...');
            const countries = config.blockedCountries || ["CN", "RU", "IN", "PK"];
            const expression = `(ip.geoip.country in {${countries.map(c => `"${c}"`).join(' ')}})`;
            try {
                await cfApi('POST', `/zones/${zoneId}/firewall/rules`, [{
                    filter: { expression, paused: false },
                    action: 'block',
                    description: 'Block Bad Countries'
                }], token);
                log('WAF Geo Block rule added.');
            } catch (e) {
                log(`WAF Error: ${e.errors?.[0]?.message || e.message}`);
            }
        }

        // Page Rules
        if (config?.pageRules) {
            log('Configuring Page Rules...');
            const addPageRule = async (target, actions) => {
                try {
                    await cfApi('POST', `/zones/${zoneId}/pagerules`, {
                        targets: [{ target: 'url', constraint: { operator: 'matches', value: target } }],
                        actions, priority: 1, status: 'active'
                    }, token);
                    log(`Page Rule: ${target}`);
                } catch (e) {
                    log(`Page Rule Error: ${e.errors?.[0]?.message || e.message}`);
                }
            };
            await addPageRule(`*${domain}/assets/*`, [{ id: 'cache_level', value: 'cache_everything' }]);
            await addPageRule(`*${domain}/admin*`, [{ id: 'security_level', value: 'high' }, { id: 'cache_level', value: 'bypass' }]);
        }

        res.json({ success: true, logs });

    } catch (error) {
        let criticalMsg = error.errors?.[0]?.message || error.message || JSON.stringify(error);
        log(`CRITICAL ERROR: ${criticalMsg}`);
        res.status(500).json({ success: false, logs, error: criticalMsg });
    }
});

// ==================== EXTERNAL API HELPERS ====================
const nicenicApi = require('./nicenic-api');
const whmApi = require('./whm-api');

// ==================== TITLECHECK ROUTES ====================
const titleCheckWebsites = require('./routes/titlecheck-websites');
const titleCheckHistory = require('./routes/titlecheck-history');
const titleCheckSettings = require('./routes/titlecheck-settings');

app.use('/api/titlecheck/websites', titleCheckWebsites);
app.use('/api/titlecheck/history', titleCheckHistory);
app.use('/api/titlecheck/settings', titleCheckSettings);

// Config file paths
const NICENIC_CONFIG_FILE = path.join(__dirname, 'data', 'nicenic.json');
const WHM_CONFIG_FILE = path.join(__dirname, 'data', 'whm.json');

// Config helpers
const getNicenicConfig = () => {
    if (!fs.existsSync(NICENIC_CONFIG_FILE)) return null;
    try {
        const content = fs.readFileSync(NICENIC_CONFIG_FILE, 'utf8');
        return content ? JSON.parse(content) : null;
    } catch (e) { return null; }
};
const saveNicenicConfig = (config) => fs.writeFileSync(NICENIC_CONFIG_FILE, JSON.stringify(config, null, 2));

const getWhmConfig = () => {
    if (!fs.existsSync(WHM_CONFIG_FILE)) return null;
    try {
        const content = fs.readFileSync(WHM_CONFIG_FILE, 'utf8');
        return content ? JSON.parse(content) : null;
    } catch (e) { return null; }
};
const saveWhmConfig = (config) => fs.writeFileSync(WHM_CONFIG_FILE, JSON.stringify(config, null, 2));

// ==================== NICENIC API ROUTES ====================

app.get('/api/nicenic/config', (req, res) => {
    const config = getNicenicConfig();
    res.json(config ? { configured: true, username: config.username } : { configured: false });
});

app.post('/api/nicenic/config', async (req, res) => {
    const { username, apiSecret } = req.body;
    if (!username || !apiSecret) return res.status(400).json({ error: 'Username and API Secret required' });

    const testResult = await nicenicApi.testConnection(username, apiSecret);
    if (!testResult.success) return res.status(400).json({ error: `Connection failed: ${testResult.error}` });

    saveNicenicConfig({ username, apiSecret, createdAt: new Date().toISOString() });
    res.json({ success: true, message: 'NiceNIC config saved' });
});

app.get('/api/nicenic/balance', async (req, res) => {
    const config = getNicenicConfig();
    if (!config) return res.status(400).json({ error: 'NiceNIC not configured' });

    try {
        const result = await nicenicApi.checkBalance(config.username, config.apiSecret);
        if (result.code === 0 || result.code === '0') {
            res.json({ success: true, balance: result.data?.amount || '0.00', currency: result.data?.currency || 'USD' });
        } else {
            res.status(400).json({ error: result.msg || 'Failed to get balance' });
        }
    } catch (e) {
        res.status(500).json({ error: e.message || 'Failed to get balance' });
    }
});

app.get('/api/nicenic/domains', async (req, res) => {
    const config = getNicenicConfig();
    if (!config) return res.status(400).json({ error: 'NiceNIC not configured' });

    try {
        const result = await nicenicApi.getDomainList(config.username, config.apiSecret);
        res.json(result);
    } catch (e) {
        res.status(500).json({ error: e.message || 'Failed to get domains' });
    }
});

app.post('/api/nicenic/redirect-to-cf', async (req, res) => {
    const { domains, cfAccountId } = req.body;
    if (!domains?.length) return res.status(400).json({ error: 'No domains specified' });
    if (!cfAccountId) return res.status(400).json({ error: 'No Cloudflare account selected' });

    const accounts = getDb();
    const cfAccount = accounts.find(a => a.id === cfAccountId);
    if (!cfAccount) return res.status(400).json({ error: 'Cloudflare account not found' });

    const nicenicConfig = getNicenicConfig();
    if (!nicenicConfig) return res.status(400).json({ error: 'NiceNIC not configured' });

    const results = [];
    for (const domain of domains) {
        try {
            // Get CF account ID
            const accountData = await cfApi('GET', '/accounts', null, cfAccount.token);
            const cloudflareAccountId = accountData.result[0].id;

            // Create zone or get existing
            let nameservers;
            try {
                const zoneData = await cfApi('POST', '/zones', {
                    name: domain, account: { id: cloudflareAccountId }, jump_start: true
                }, cfAccount.token);
                nameservers = zoneData.result.name_servers;
            } catch (e) {
                if (e.errors?.[0]?.message?.includes('already exists')) {
                    const zones = await cfApi('GET', `/zones?name=${domain}`, null, cfAccount.token);
                    nameservers = zones.result?.[0]?.name_servers;
                } else throw e;
            }

            if (!nameservers?.length) throw new Error('Could not get nameservers');

            // Update NiceNIC
            await nicenicApi.updateNameservers(domain, nameservers[0], nameservers[1], nicenicConfig.username, nicenicConfig.apiSecret);
            results.push({ domain, success: true, nameservers });
        } catch (e) {
            results.push({ domain, success: false, error: e.errors?.[0]?.message || e.message });
        }
    }

    res.json({ success: results.every(r => r.success), message: `${results.filter(r => r.success).length}/${domains.length} processed`, results });
});

// Update nameservers for multiple domains
app.post('/api/nicenic/update-ns', async (req, res) => {
    const { domains, nameservers } = req.body;

    // Validate input
    if (!domains?.length) {
        return res.status(400).json({ error: 'No domains specified' });
    }
    if (!nameservers?.length || nameservers.length < 2) {
        return res.status(400).json({ error: 'At least 2 nameservers required' });
    }

    const config = getNicenicConfig();
    if (!config) {
        return res.status(400).json({ error: 'NiceNIC not configured' });
    }

    const results = [];
    for (const domain of domains) {
        try {
            // NiceNIC API requires at least dns1 and dns2
            await nicenicApi.updateNameservers(
                domain,
                nameservers[0],
                nameservers[1],
                config.username,
                config.apiSecret
            );
            results.push({ domain, success: true });
        } catch (e) {
            results.push({ domain, success: false, error: e.message || 'Update failed' });
        }
    }

    res.json({
        success: results.every(r => r.success),
        message: `${results.filter(r => r.success).length}/${domains.length} updated`,
        results
    });
});

// ==================== WHM API ROUTES ====================

app.get('/api/whm/config', (req, res) => {
    const config = getWhmConfig();
    res.json(config ? { configured: true, hostname: config.hostname, port: config.port } : { configured: false });
});

app.post('/api/whm/config', async (req, res) => {
    const { hostname, port, apiToken } = req.body;
    if (!hostname || !apiToken) return res.status(400).json({ error: 'Hostname and API Token required' });

    const config = { hostname, port: port || 2087, apiToken };
    const testResult = await whmApi.testConnection(config);
    if (!testResult.success) return res.status(400).json({ error: `Connection failed: ${testResult.error}` });

    saveWhmConfig({ ...config, createdAt: new Date().toISOString() });
    res.json({ success: true, message: 'WHM config saved', version: testResult.version });
});

app.get('/api/whm/packages', async (req, res) => {
    const config = getWhmConfig();
    if (!config) return res.status(400).json({ error: 'WHM not configured' });

    try {
        const result = await whmApi.listPackages(config);
        res.json(result.package || []);
    } catch (e) {
        res.status(500).json({ error: e.message });
    }
});

app.post('/api/whm/create-account', async (req, res) => {
    const { domain, username, password, plan, subdomains } = req.body;
    if (!domain) return res.status(400).json({ error: 'Domain required' });

    const config = getWhmConfig();
    if (!config) return res.status(400).json({ error: 'WHM not configured' });

    try {
        const accountResult = await whmApi.createAccount({ domain, username, password, plan }, config);
        const results = { account: accountResult, subdomains: [] };

        if (subdomains?.length) {
            for (const sub of subdomains) {
                if (!sub.trim() || sub === 'www') continue;
                try {
                    await whmApi.addSubdomain(sub.trim(), domain, accountResult.username, config);
                    results.subdomains.push({ subdomain: sub, success: true });
                } catch (e) {
                    results.subdomains.push({ subdomain: sub, success: false, error: e.message });
                }
            }
        }

        res.json({ success: true, ...results });
    } catch (e) {
        res.status(500).json({ error: e.message });
    }

});

// ==================== INDEXER API ROUTES ====================

app.get('/api/indexer/profile', async (req, res) => {
    try {
        const response = await axios.get('https://hm-api.hacklink.market/api/user/profile', {
            headers: {
                'Content-Type': 'application/json',
                'X-API-Key': '966683ed00f7df381ec5ed708014ca08b64d2454bdde4e2215d704b5a37524cf'
            }
        });
        res.json(response.data);
    } catch (e) {
        console.error('Indexer Profile Error:', e.message);
        res.status(e.response?.status || 500).json({ error: e.message });
    }
});

app.post('/api/indexer/submit', async (req, res) => {
    const { website } = req.body;
    if (!website) return res.status(400).json({ error: 'Website URL required' });

    try {
        const response = await axios.post('https://hm-api.hacklink.market/api/user/indexer/submit', {
            website
        }, {
            headers: {
                'Content-Type': 'application/json',
                'X-API-Key': '966683ed00f7df381ec5ed708014ca08b64d2454bdde4e2215d704b5a37524cf'
            }
        });

        // Check if response is HTML (Login Page)
        if (typeof response.data === 'string' && response.data.trim().startsWith('<!DOCTYPE html>')) {
            throw new Error('API request rejected (Redirected to Login). Check your API Key or Endpoint URL.');
        }

        console.log('Indexer API Response:', response.data);
        res.json(response.data);
    } catch (e) {
        const isHtmlError = typeof e.response?.data === 'string' && e.response?.data.includes('<!DOCTYPE html>');

        console.error('Indexer API Error Details:', {
            message: e.message,
            response: isHtmlError ? 'HTML Login Page Detected' : e.response?.data,
            status: e.response?.status
        });

        let errorMsg = e.response?.data?.error || e.response?.data?.message || e.message || 'Bilinmeyen bir hata oluştu';
        if (isHtmlError || e.message.includes('Redirected to Login')) {
            errorMsg = 'API Yetkilendirme Hatası: Sunucu "Giriş Yap" sayfasına yönlendiriyor. API Key geçersiz olabilir veya bu işlem için oturum açmak gerekebilir.';
        }

        res.status(e.response?.status || 500).json({
            success: false,
            error: errorMsg,
            message: errorMsg,
            details: e.response?.data
        });
    }
});


// Upload and extract ZIP file to cPanel account
app.post('/api/whm/upload', upload.single('file'), async (req, res) => {
    const { username, domain, subdomain, password } = req.body;

    if (!req.file && !req.body.designFile) {
        return res.status(400).json({ error: 'No file uploaded or design selected' });
    }

    if (!username && !domain) {
        return res.status(400).json({ error: 'Username or domain required' });
    }

    const config = getWhmConfig();
    if (!config) return res.status(400).json({ error: 'WHM not configured' });

    // Add password to config for this request if provided
    if (password) {
        config.cpanelPassword = password;
    }

    try {
        const targetUsername = username || domain.replace(/\./g, '').substring(0, 8).toLowerCase();

        // Determine destination directory
        let destDir = '/public_html';
        if (subdomain && subdomain !== domain) {
            // If subdomain, upload to subdomain directory (Root level as per user config)
            destDir = `/${subdomain}`;
        }

        // Determine filename and buffer
        let filename, fileBuffer;

        if (req.body.designFile) {
            // Use stored design
            filename = req.body.designFile;
            const designPath = path.join(DESIGNS_DIR, filename);
            if (!fs.existsSync(designPath)) {
                throw new Error('Selected design file not found in repository');
            }
            fileBuffer = fs.readFileSync(designPath);
            console.log(`[WHM Upload] Using stored design: ${filename}`);
        } else {
            // Use uploaded file
            filename = req.file.originalname.replace(/\s+/g, '_');
            fileBuffer = req.file.buffer;
        }

        // If it's a ZIP, try to extract it
        if (filename.endsWith('.zip') || filename.endsWith('.rar')) {
            console.log(`[WHM Upload] Uploading and Extracting...`);

            // 1. Upload File first
            await whmApi.uploadFile(targetUsername, fileBuffer, filename, destDir, config);

            // 2. Extract
            const zipPath = `${destDir}/${filename}`;
            try {
                await whmApi.extractZip(targetUsername, zipPath, destDir, config);
                res.json({
                    success: true,
                    message: 'File uploaded and extracted',
                    destination: destDir
                });
            } catch (extractErr) {
                res.json({
                    success: true,
                    message: 'File uploaded but extraction failed',
                    destination: destDir,
                    extractError: extractErr.message
                });
            }
        } else {
            await whmApi.uploadFile(targetUsername, fileBuffer, filename, destDir, config);
            res.json({
                success: true,
                message: 'File uploaded',
                destination: destDir
            });
        }
    } catch (e) {
        console.error('[WHM Upload] Error:', e);
        res.status(500).json({ error: e.message || 'Upload failed' });
    }
});

// ==================== DOMAIN CHECKER ====================

app.post('/api/domain-check', async (req, res) => {
    const { domain } = req.body;
    if (!domain) return res.status(400).json({ error: 'Domain required' });

    const config = getNicenicConfig();
    if (!config) return res.status(400).json({ error: 'NiceNIC not configured' });

    try {
        const result = await nicenicApi.checkDomainAvailability(domain, config.username, config.apiSecret);
        const availability = result.data?.[domain] || 'error';
        res.json({ domain, available: availability === 'available', status: availability });
    } catch (e) {
        res.status(500).json({ error: e.message });
    }
});

app.post('/api/domain-bulk-check', async (req, res) => {
    const { keyword, tld = '.com', format = 'prefix', start = 1, end = 50, maxResults = 5 } = req.body;
    if (!keyword) return res.status(400).json({ error: 'Keyword required' });

    const config = getNicenicConfig();
    if (!config) return res.status(400).json({ error: 'NiceNIC not configured' });

    const results = [];
    const available = [];

    for (let i = start; i <= end && available.length < maxResults; i++) {
        const domain = format === 'prefix'
            ? `${i}${keyword}${tld}`
            : `${keyword}${i}${tld}`;

        try {
            const result = await nicenicApi.checkDomainAvailability(domain, config.username, config.apiSecret);
            const status = result.data?.[domain] || 'error';
            const isAvailable = status === 'available';

            results.push({ domain, available: isAvailable, status });
            if (isAvailable) available.push(domain);

            console.log(`[Domain Check] ${domain}: ${status}`);
        } catch (e) {
            results.push({ domain, available: false, status: 'error', error: e.message });
        }

        // Small delay to avoid rate limiting
        await new Promise(r => setTimeout(r, 200));
    }

    res.json({
        results,
        availableCount: available.length,
        available,
        stopped: available.length >= maxResults
    });
});

// ==================== WHOIS CONFIG ====================

const WHOIS_CONFIG_FILE = path.join(__dirname, 'data', 'whois.json');

const getWhoisConfig = () => {
    if (!fs.existsSync(WHOIS_CONFIG_FILE)) {
        // Default WHOIS info
        return {
            organization: 'Private Registration',
            name: 'Domain Admin',
            country: 'TR',
            state: 'Istanbul',
            city: 'Istanbul',
            address: 'Istanbul, Turkey',
            postcode: '34000',
            phone: '+90.5000000000',
            fax: '+90.5000000000',
            email: 'admin@domain.com'
        };
    }
    try {
        return JSON.parse(fs.readFileSync(WHOIS_CONFIG_FILE, 'utf8'));
    } catch (e) { return null; }
};

const saveWhoisConfig = (config) => fs.writeFileSync(WHOIS_CONFIG_FILE, JSON.stringify(config, null, 2));

app.get('/api/whois/config', (req, res) => {
    res.json(getWhoisConfig());
});

app.post('/api/whois/config', (req, res) => {
    const whois = req.body;
    saveWhoisConfig(whois);
    res.json({ success: true, message: 'WHOIS config saved' });
});

// ==================== DESIGN REPOSITORY ====================

app.get('/api/designs', (req, res) => {
    try {
        const files = fs.readdirSync(DESIGNS_DIR).filter(f => f.endsWith('.zip') || f.endsWith('.rar'));

        // ==================== DESIGN REPOSITORY (Continued) ====================
        const stats = files.map(f => {
            const st = fs.statSync(path.join(DESIGNS_DIR, f));
            return { name: f, size: st.size, date: st.mtime };
        });
        res.json(stats);
    } catch (e) {
        res.status(500).json({ error: e.message });
    }
});

app.post('/api/designs', uploadDesign.single('file'), (req, res) => {
    if (!req.file) return res.status(400).json({ error: 'No file uploaded' });
    res.json({ success: true, message: 'Design uploaded', filename: req.file.filename });
});

app.delete('/api/designs/:filename', (req, res) => {
    const filePath = path.join(DESIGNS_DIR, req.params.filename);
    if (fs.existsSync(filePath)) {
        fs.unlinkSync(filePath);
        res.json({ success: true });
    } else {
        res.status(404).json({ error: 'File not found' });
    }
});

// ==================== CLICKAINE CAMPAIGN ROUTES ====================

// Get all campaigns status
app.get('/api/campaigns', async (req, res) => {
    const result = await clickaineApiRequest('GET', '/v1/public/advertiser/campaigns?limit=100');
    res.json(result);
});

// Get single campaign
app.get('/api/campaigns/:id', async (req, res) => {
    const result = await clickaineApiRequest('GET', `/v1/public/advertiser/campaigns?ids=${req.params.id}`);
    res.json(result);
});

// Create campaign
app.post('/api/campaigns', async (req, res) => {
    const { sourceUrl, destUrl } = req.body;
    const urls = [sourceUrl];
    if (destUrl) urls.push(destUrl);

    // Create creatives
    const creativeIds = [];
    for (let i = 0; i < urls.length; i++) {
        const result = await clickaineApiRequest('POST', '/v1/public/advertiser/creatives', {
            adFormat: CAMPAIGN_TEMPLATE.adFormat,
            themeID: CAMPAIGN_TEMPLATE.themeID,
            clickURL: urls[i],
            url: urls[i],
            name: `Web Creative ${i + 1}`
        });
        if (result.success && result.data.id) {
            creativeIds.push(result.data.id);
        } else {
            return res.json({ success: false, error: `Creative ${i + 1} failed: ${JSON.stringify(result.error)}` });
        }
    }

    // Copy template campaign
    const copyResult = await clickaineApiRequest('GET', `/v1/public/advertiser/campaigns/${TEMPLATE_CAMPAIGN_ID}/copy`);
    if (!copyResult.success || !copyResult.data.id) {
        return res.json({ success: false, error: 'Template copy failed' });
    }

    const newCampaignId = copyResult.data.id;

    // Update with new creatives
    const updateResult = await clickaineApiRequest('PATCH', `/v1/public/advertiser/campaign/${newCampaignId}`, {
        name: generateCampaignName(),
        overallBudget: CAMPAIGN_TEMPLATE.overallBudget,
        target: { bid: CAMPAIGN_TEMPLATE.bid, creatives: creativeIds.map(id => ({ id })) }
    });

    res.json(updateResult.success
        ? { success: true, data: { id: newCampaignId, creatives: creativeIds.length } }
        : { success: false, error: updateResult.error });
});

// Start campaign (Single)
app.post('/api/campaigns/:id/start', async (req, res) => {
    const result = await clickaineApiRequest('GET', `/v1/public/advertiser/campaigns/${req.params.id}/start`);
    res.json(result);
});

// Batch Start
app.post('/api/campaigns/batch/start', async (req, res) => {
    const { ids } = req.body; // Expects comma separated string or array
    const idList = Array.isArray(ids) ? ids.join(',') : ids;
    const result = await clickaineApiRequest('POST', `/v1/public/advertiser/campaigns/batch-start?ids=${idList}`);
    res.json(result);
});

// Stop campaign (Single)
app.post('/api/campaigns/:id/stop', async (req, res) => {
    const result = await clickaineApiRequest('GET', `/v1/public/advertiser/campaigns/${req.params.id}/stop`);
    res.json(result);
});

// Batch Stop
app.post('/api/campaigns/batch/stop', async (req, res) => {
    const { ids } = req.body;
    const idList = Array.isArray(ids) ? ids.join(',') : ids;
    const result = await clickaineApiRequest('POST', `/v1/public/advertiser/campaigns/batch-stop?ids=${idList}`);
    res.json(result);
});

// Delete campaign (Single or Batch via query)
app.delete('/api/campaigns/:id', async (req, res) => {
    // Determine if :id is a comma separated list or single
    const result = await clickaineApiRequest('DELETE', `/v1/public/advertiser/campaigns?ids=${req.params.id}`);
    res.json(result);
});

// Batch Delete (Explicit)
app.post('/api/campaigns/batch/delete', async (req, res) => {
    const { ids } = req.body;
    const idList = Array.isArray(ids) ? ids.join(',') : ids;
    const result = await clickaineApiRequest('DELETE', `/v1/public/advertiser/campaigns?ids=${idList}`);
    res.json(result);
});

// Get campaign URLs
app.get('/api/campaigns/:id/urls', async (req, res) => {
    const campaignId = req.params.id;

    // Get campaign details to find creatives
    const campaignResult = await clickaineApiRequest('GET', `/v1/public/advertiser/campaigns?ids=${campaignId}`);
    if (!campaignResult.success || !campaignResult.data.campaigns?.[0]) {
        return res.json({ success: false, error: 'Campaign not found' });
    }

    const campaign = campaignResult.data.campaigns[0];

    // Prefer target.creatives as it contains the active configuration
    // The top-level 'creatives' field might be stale or historical
    let creatives = campaign.target?.creatives || campaign.creatives;

    if (!creatives || creatives.length === 0) {
        return res.json({ success: true, data: [] });
    }

    // Ensure we have a list of IDs (handle partial object case if API changes)
    creatives = creatives.map(c => (typeof c === 'object' && c.id) ? c.id : c);

    // Fetch details for each creative
    const urls = [];
    for (const creativeId of creatives) {
        const creativeResult = await clickaineApiRequest('GET', `/v1/public/advertiser/creatives/${creativeId}`);
        if (creativeResult.success) {
            urls.push(creativeResult.data.clickURL || creativeResult.data.url);
        }
    }

    res.json({ success: true, data: urls });
});

// Update campaign URLs
app.put('/api/campaigns/:id/urls', async (req, res) => {
    const { sourceUrl, destUrl } = req.body;
    const campaignId = req.params.id;

    // Get old creatives
    const campaignResult = await clickaineApiRequest('GET', `/v1/public/advertiser/campaigns?ids=${campaignId}`);
    if (!campaignResult.success || !campaignResult.data.campaigns?.[0]) {
        return res.json({ success: false, error: 'Campaign not found' });
    }

    const oldCreatives = campaignResult.data.campaigns[0].creatives || [];

    // Delete old creatives
    for (const id of oldCreatives) {
        await clickaineApiRequest('DELETE', `/v1/public/advertiser/creatives/${id}`);
    }

    // Create new creatives
    const urls = [sourceUrl];
    if (destUrl) urls.push(destUrl);

    const newIds = [];
    for (let i = 0; i < urls.length; i++) {
        const result = await clickaineApiRequest('POST', '/v1/public/advertiser/creatives', {
            adFormat: CAMPAIGN_TEMPLATE.adFormat,
            themeID: CAMPAIGN_TEMPLATE.themeID,
            clickURL: urls[i],
            url: urls[i],
            name: `Web Creative ${i + 1}`
        });
        if (result.success && result.data.id) {
            newIds.push(result.data.id);
        } else {
            return res.json({ success: false, error: 'Creative creation failed' });
        }
    }

    // Update campaign
    const result = await clickaineApiRequest('PATCH', `/v1/public/advertiser/campaign/${campaignId}`, {
        target: { creatives: newIds.map(id => ({ id })) }
    });

    res.json(result);
});

// Update budget (INDEX)
app.put('/api/campaigns/:id/budget', async (req, res) => {
    const { budget } = req.body;
    const result = await clickaineApiRequest('PATCH', `/v1/public/advertiser/campaign/${req.params.id}`, {
        overallBudget: parseFloat(budget)
    });
    res.json(result);
});

// Get balance
app.get('/api/balance', async (req, res) => {
    const result = await clickaineApiRequest('GET', '/v1/public/finance/balance');
    res.json(result);
});

// ==================== DOMAIN PURCHASE ====================

app.post('/api/nicenic/purchase', async (req, res) => {
    const { domain, years = 1 } = req.body;
    if (!domain) return res.status(400).json({ error: 'Domain required' });

    const config = getNicenicConfig();
    if (!config) return res.status(400).json({ error: 'NiceNIC not configured' });

    const whois = getWhoisConfig();

    try {
        // First verify availability
        const checkResult = await nicenicApi.checkDomainAvailability(domain, config.username, config.apiSecret);
        const status = checkResult.data?.[domain];

        if (status !== 'available') {
            return res.status(400).json({ error: `Domain not available: ${status}` });
        }

        // Purchase the domain
        console.log(`[NiceNIC] Purchasing domain: ${domain}`);
        const result = await nicenicApi.registerDomain(domain, whois, years, config.username, config.apiSecret);

        if (result.code === 0 || result.code === '0') {
            res.json({
                success: true,
                message: 'Domain purchased successfully',
                domain: result.data?.domain || domain,
                expires: result.data?.expired_date
            });
        } else {
            res.status(400).json({ error: result.msg || 'Purchase failed' });
        }
    } catch (e) {
        console.error('[NiceNIC] Purchase error:', e);
        res.status(500).json({ error: e.msg || e.message || 'Purchase failed' });
    }
});

// ==================== GENERIC API 404 HANDLER ====================
app.use('/api', (req, res) => {
    console.log(`[404] Route not found: ${req.method} ${req.path}`);
    res.status(404).json({ error: `Route not found: ${req.method} ${req.path}` });
});

// Start Server
app.listen(PORT, async () => {
    console.log(`\n╔════════════════════════════════════════╗`);
    console.log(`║  Cloudflare Automator PRO Started      ║`);
    console.log(`╚════════════════════════════════════════╝\n`);
    console.log(`🌐 Server running on: http://localhost:${PORT}`);

    // Initialize TitleCheck data directory
    const titleCheckDataDir = path.join(__dirname, 'data', 'titlecheck');
    if (!fs.existsSync(titleCheckDataDir)) {
        fs.mkdirSync(titleCheckDataDir, { recursive: true });
        console.log('[TitleCheck] Created data directory');
    }

    // Start TitleCheck scheduler
    const titleCheckScheduler = require('./services/scheduler');
    const titleChecker = require('./services/titleChecker');
    await titleCheckScheduler.start();

    // Graceful shutdown handlers
    const gracefulShutdown = async () => {
        console.log('\n🛑 Shutting down gracefully...');
        titleCheckScheduler.stop();
        await titleChecker.cleanup();
        process.exit(0);
    };

    process.on('SIGTERM', gracefulShutdown);
    process.on('SIGINT', gracefulShutdown);
});
