/**
 * NiceNIC API V2 Helper
 * Based on official documentation
 * 
 * HTTP Endpoint: http://api.nicenic.net/v2/
 * Auth: Authorization header with username:api_secret
 * Format: Query string parameters (not JSON body)
 */

const axios = require('axios');

const NICENIC_API_BASE = 'http://api.nicenic.net/v2/';

/**
 * Make a NiceNIC API request
 */
async function nicenicApi(category, action, params = {}, username, apiSecret) {
    try {
        const queryParams = new URLSearchParams({
            category,
            action,
            ...params
        });

        const url = `${NICENIC_API_BASE}?${queryParams.toString()}`;

        console.log(`[NiceNIC API] ${category}/${action} - URL: ${url.substring(0, 120)}...`);

        const response = await axios.get(url, {
            headers: {
                'Authorization': `${username}:${apiSecret}`,
                'Content-Type': 'application/json',
                'Host': 'api.nicenic.net'
            }
        });

        console.log(`[NiceNIC API] Response code:`, response.data?.code);
        return response.data;
    } catch (error) {
        console.error(`NiceNIC API Error [${category}/${action}]:`, error.response?.data || error.message);
        throw error.response?.data || error;
    }
}

/**
 * Get list of all domains in account
 */
async function getDomainList(username, apiSecret, options = {}) {
    const params = {
        page_size: options.pageSize || 100,
        page_index: options.pageIndex || 1,
        status: options.status || 'all',
        sort: options.sort || 'time_desc'
    };

    const result = await nicenicApi('domain', 'domain_list', params, username, apiSecret);
    return result;
}

/**
 * Get detailed info for a specific domain
 */
async function getDomainInfo(domain, username, apiSecret) {
    const result = await nicenicApi('domain', 'info', { domain }, username, apiSecret);
    return result;
}

/**
 * Update nameservers for a domain
 */
async function updateNameservers(domain, dns1, dns2, username, apiSecret) {
    const result = await nicenicApi('domain', 'change_dns', {
        domain,
        dns1,
        dns2
    }, username, apiSecret);
    return result;
}

/**
 * Check domain availability - supports multiple domains comma-separated
 */
async function checkDomainAvailability(domains, username, apiSecret) {
    const domainsStr = Array.isArray(domains) ? domains.join(',') : domains;
    const result = await nicenicApi('domain', 'check', { domains: domainsStr }, username, apiSecret);
    return result;
}

/**
 * Register (purchase) a domain
 * @param {string} domain - Domain name to register
 * @param {object} whoisInfo - WHOIS registrant info
 * @param {number} years - Registration period (1-10)
 */
async function registerDomain(domain, whoisInfo, years = 1, username, apiSecret) {
    const params = {
        domain,
        year: years,
        dns1: 'ns3.my-ndns.com',  // Default NiceNIC nameservers
        dns2: 'ns4.my-ndns.com',
        organization: whoisInfo.organization || 'Private',
        name: whoisInfo.name || 'Domain Admin',
        country: whoisInfo.country || 'TR',
        state: whoisInfo.state || 'Istanbul',
        city: whoisInfo.city || 'Istanbul',
        address: whoisInfo.address || 'Istanbul',
        postcode: whoisInfo.postcode || '34000',
        phone: whoisInfo.phone || '+90.5000000000',
        fax: whoisInfo.fax || '+90.5000000000',
        email: whoisInfo.email || 'admin@domain.com'
    };

    console.log(`[NiceNIC] Registering domain: ${domain} for ${years} year(s)`);
    const result = await nicenicApi('domain', 'register', params, username, apiSecret);
    return result;
}

/**
 * Test API connection by trying to get domain list
 */
async function testConnection(username, apiSecret) {
    try {
        const result = await getDomainList(username, apiSecret);
        if (result && (result.code === 0 || result.code === '0' || result.data)) {
            return { success: true, result };
        }
        return { success: false, error: result.msg || 'Unknown error' };
    } catch (error) {
        return { success: false, error: error.message || 'Connection failed' };
    }
}

/**
 * Check account balance
 */
async function checkBalance(username, apiSecret) {
    // API V2 uses category=user & action=balance
    const result = await nicenicApi('user', 'balance', {}, username, apiSecret);
    return result;
}

module.exports = {
    nicenicApi,
    getDomainList,
    getDomainInfo,
    updateNameservers,
    checkDomainAvailability,
    registerDomain,
    testConnection,
    checkBalance
};
