// API Base URL
const API_URL = '/api/titlecheck';

// State
let websites = [];
let history = [];
let settings = {};
const authToken = sessionStorage.getItem('authToken');

// Initialize
document.addEventListener('DOMContentLoaded', () => {
    initEventListeners();
    loadSettings();
    loadWebsites();
    loadHistory();

    // Auto-refresh history every 30 seconds
    setInterval(loadHistory, 30000);
});

// Event Listeners
function initEventListeners() {
    // Add website form
    document.getElementById('addWebsiteForm').addEventListener('submit', handleAddWebsite);

    // Settings form
    document.getElementById('settingsForm').addEventListener('submit', handleSaveSettings);

    // Close modal on backdrop click
    document.getElementById('settingsModal').addEventListener('click', (e) => {
        if (e.target.id === 'settingsModal') {
            closeSettings();
        }
    });
}

// API Calls
async function apiCall(endpoint, options = {}) {
    try {
        const response = await fetch(`${API_URL}${endpoint}`, {
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${authToken}`,
                ...options.headers
            },
            ...options
        });

        const data = await response.json();

        if (!response.ok) {
            throw new Error(data.error || 'Bir hata oluştu');
        }

        return data;
    } catch (error) {
        console.error('API Error:', error);
        showToast(error.message, 'error');
        throw error;
    }
}

// Load Data
async function loadWebsites() {
    try {
        const data = await apiCall('/websites');
        websites = data.websites;
        renderWebsites();
    } catch (error) {
        console.error('Error loading websites:', error);
    }
}

async function loadHistory() {
    try {
        const data = await apiCall('/history?limit=50');
        history = data.history;
        renderHistory();
    } catch (error) {
        console.error('Error loading history:', error);
    }
}

async function loadSettings() {
    try {
        const data = await apiCall('/settings');
        settings = data.settings;
    } catch (error) {
        console.error('Error loading settings:', error);
    }
}

// Add Website
async function handleAddWebsite(e) {
    e.preventDefault();

    const url = document.getElementById('websiteUrl').value.trim();
    const note = document.getElementById('websiteNote').value.trim();

    try {
        showToast('Website ekleniyor... (İlk kontrol yapılıyor)', 'success');

        await apiCall('/websites', {
            method: 'POST',
            body: JSON.stringify({ url, note })
        });

        document.getElementById('addWebsiteForm').reset();
        await loadWebsites();
        await loadHistory();
        showToast('Website başarıyla eklendi', 'success');
    } catch (error) {
        // Error already shown in apiCall
    }
}

// Delete Website
async function deleteWebsite(id) {
    if (!confirm('Bu website\'i silmek istediğinizden emin misiniz?')) {
        return;
    }

    try {
        await apiCall(`/websites/${id}`, {
            method: 'DELETE'
        });

        await loadWebsites();
        showToast('Website silindi', 'success');
    } catch (error) {
        // Error already shown in apiCall
    }
}

// Check Website
async function checkWebsite(id) {
    try {
        const button = event.target;
        const originalText = button.innerHTML;
        button.disabled = true;
        button.innerHTML = '<i class="fa-solid fa-spinner fa-spin"></i> Kontrol ediliyor...';

        await apiCall(`/websites/${id}/check`, {
            method: 'POST'
        });

        await loadWebsites();
        await loadHistory();
        showToast('Kontrol tamamlandı', 'success');
    } catch (error) {
        // Error already shown in apiCall
    } finally {
        const button = event.target;
        button.disabled = false;
        button.innerHTML = '<i class="fa-solid fa-refresh"></i> Kontrol Et';
    }
}

// Settings
function openSettings() {
    document.getElementById('settingsModal').classList.add('active');

    // Populate form
    document.getElementById('checkInterval').value = settings.checkInterval || 15;
    document.getElementById('historyLimit').value = settings.historyLimit || 1000;
    document.getElementById('screenshotEnabled').checked = settings.screenshotEnabled !== false;
    document.getElementById('telegramToken').value = settings.telegramToken || '';
    document.getElementById('telegramChatId').value = settings.telegramChatId || '';
}

function closeSettings() {
    document.getElementById('settingsModal').classList.remove('active');
}

async function handleSaveSettings(e) {
    e.preventDefault();

    const checkInterval = parseInt(document.getElementById('checkInterval').value);
    const historyLimit = parseInt(document.getElementById('historyLimit').value);
    const screenshotEnabled = document.getElementById('screenshotEnabled').checked;
    const telegramToken = document.getElementById('telegramToken').value.trim();
    const telegramChatId = document.getElementById('telegramChatId').value.trim();

    try {
        const data = await apiCall('/settings', {
            method: 'PUT',
            body: JSON.stringify({
                checkInterval,
                historyLimit,
                screenshotEnabled,
                telegramToken,
                telegramChatId
            })
        });

        settings = data.settings;
        closeSettings();
        showToast('Ayarlar kaydedildi ve scheduler güncellendi', 'success');
    } catch (error) {
        // Error already shown in apiCall
    }
}

// Render Functions
function renderWebsites() {
    const container = document.getElementById('websitesList');
    const countBadge = document.getElementById('websiteCount');

    countBadge.textContent = websites.length;

    if (websites.length === 0) {
        container.innerHTML = `
            <div class="empty-state">
                <i class="fa-solid fa-inbox" style="font-size: 3rem; margin-bottom: 1rem; opacity: 0.3;"></i>
                <p>Henüz website eklemediniz</p>
            </div>
        `;
        return;
    }

    const html = websites.map(website => `
        <div class="website-card">
            <div class="website-url">${escapeHtml(website.url)}</div>
            ${website.note ? `<div class="website-note">${escapeHtml(website.note)}</div>` : ''}
            
            ${website.lastTitle ? `
                <div class="website-title-box">
                    <strong>Son Title</strong>
                    <div>${escapeHtml(website.lastTitle)}</div>
                </div>
            ` : '<div class="website-title-box"><div style="color: #888;">Henüz kontrol edilmedi</div></div>'}
            
            <div class="website-meta">
                <span>${website.lastCheck ? formatDate(website.lastCheck) : 'Hiç kontrol edilmedi'}</span>
                ${website.lastStatusCode ? `<span>Status: ${website.lastStatusCode}</span>` : ''}
            </div>
            
            <div class="website-actions">
                <button class="btn-check" onclick="checkWebsite('${website.id}')">
                    <i class="fa-solid fa-refresh"></i> Kontrol Et
                </button>
                <button class="btn-delete" onclick="deleteWebsite('${website.id}')">
                    <i class="fa-solid fa-trash"></i> Sil
                </button>
            </div>
        </div>
    `).join('');

    container.innerHTML = html;
}

function renderHistory() {
    const container = document.getElementById('historyList');

    if (history.length === 0) {
        container.innerHTML = '<div class="empty-state"><p>Henüz kontrol geçmişi yok</p></div>';
        return;
    }

    const html = history.map(item => `
        <div class="history-item ${item.titleChanged ? 'changed' : ''}">
            <div class="history-header">
                <div>
                    <div class="history-url">${escapeHtml(item.websiteUrl)}</div>
                    ${item.websiteNote ? `<div style="color: #888; font-size: 0.85rem;">${escapeHtml(item.websiteNote)}</div>` : ''}
                </div>
                <div class="history-time">${formatDate(item.timestamp)}</div>
            </div>
            
            ${item.success ? `
                <div class="history-title">${escapeHtml(item.title)}</div>
                
                ${item.titleChanged ? `
                    <div class="title-change">
                        <div style="font-size: 0.75rem; color: #ff9800; font-weight: 600; margin-bottom: 0.5rem;">⚠️ TITLE DEĞİŞTİ</div>
                        <div class="title-old">Eski: ${escapeHtml(item.oldTitle)}</div>
                        <div class="title-new">Yeni: ${escapeHtml(item.title)}</div>
                    </div>
                ` : ''}
            ` : `
                <div class="history-title" style="color: #f44336;">❌ Kontrol başarısız</div>
            `}
        </div>
    `).join('');

    container.innerHTML = html;
}

// Utility Functions
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

function formatDate(dateString) {
    const date = new Date(dateString);
    const now = new Date();
    const diff = now - date;

    const minutes = Math.floor(diff / 60000);
    const hours = Math.floor(minutes / 60);
    const days = Math.floor(hours / 24);

    if (minutes < 1) return 'Az önce';
    if (minutes < 60) return `${minutes} dakika önce`;
    if (hours < 24) return `${hours} saat önce`;
    if (days < 7) return `${days} gün önce`;

    return date.toLocaleDateString('tr-TR', {
        year: 'numeric',
        month: 'long',
        day: 'numeric',
        hour: '2-digit',
        minute: '2-digit'
    });
}

function showToast(message, type = 'success') {
    const toast = document.getElementById('toast');
    toast.textContent = message;
    toast.className = `toast ${type} show`;

    setTimeout(() => {
        toast.classList.remove('show');
    }, 3000);
}
