const express = require('express');
const router = express.Router();
const fs = require('fs').promises;
const path = require('path');
const scheduler = require('../services/scheduler');
const telegram = require('../services/telegram');

const settingsPath = path.join(__dirname, '../data/titlecheck/settings.json');

// Get current settings
router.get('/', async (req, res) => {
    try {
        const data = await fs.readFile(settingsPath, 'utf8');
        const settings = JSON.parse(data);
        res.json({ success: true, settings });
    } catch (error) {
        // Return default settings if file doesn't exist
        res.json({
            success: true,
            settings: {
                telegramToken: '',
                telegramChatId: '',
                checkInterval: 15,
                historyLimit: 1000,
                screenshotEnabled: true
            }
        });
    }
});

// Update settings
router.put('/', async (req, res) => {
    try {
        const { telegramToken, telegramChatId, checkInterval, historyLimit, screenshotEnabled } = req.body;

        // Load current settings
        let settings;
        try {
            const data = await fs.readFile(settingsPath, 'utf8');
            settings = JSON.parse(data);
        } catch (error) {
            settings = {
                telegramToken: '',
                telegramChatId: '',
                checkInterval: 15,
                historyLimit: 1000,
                screenshotEnabled: true
            };
        }

        // Update settings
        if (telegramToken !== undefined) settings.telegramToken = telegramToken;
        if (telegramChatId !== undefined) settings.telegramChatId = telegramChatId;
        if (checkInterval !== undefined) settings.checkInterval = parseInt(checkInterval);
        if (historyLimit !== undefined) settings.historyLimit = parseInt(historyLimit);
        if (screenshotEnabled !== undefined) settings.screenshotEnabled = screenshotEnabled;

        // Ensure directory exists
        const dataDir = path.dirname(settingsPath);
        await fs.mkdir(dataDir, { recursive: true });

        // Save settings
        await fs.writeFile(settingsPath, JSON.stringify(settings, null, 2));

        // Update telegram credentials if changed
        if (telegramToken || telegramChatId) {
            await telegram.updateCredentials(
                settings.telegramToken,
                settings.telegramChatId
            );
        }

        // Update scheduler interval if changed
        if (checkInterval) {
            await scheduler.updateInterval(settings.checkInterval);
        }

        res.json({ success: true, settings });
    } catch (error) {
        res.status(500).json({ success: false, error: error.message });
    }
});

module.exports = router;
