const express = require('express');
const router = express.Router();
const titleChecker = require('../services/titleChecker');
const telegram = require('../services/telegram');

// Get all websites
router.get('/', async (req, res) => {
    try {
        const websites = await titleChecker.loadWebsites();
        res.json({ success: true, websites });
    } catch (error) {
        res.status(500).json({ success: false, error: error.message });
    }
});

// Add new website
router.post('/', async (req, res) => {
    try {
        const { url, note } = req.body;

        if (!url) {
            return res.status(400).json({ success: false, error: 'URL is required' });
        }

        console.log(`[TitleCheck] Adding new website: ${url}`);
        const websites = await titleChecker.loadWebsites();

        const newWebsite = {
            id: titleChecker.generateId(),
            url: url.trim(),
            note: note ? note.trim() : '',
            lastTitle: null,
            lastStatusCode: null,
            lastLoadTime: null,
            lastCheck: null,
            createdAt: new Date().toISOString()
        };

        websites.push(newWebsite);
        await titleChecker.saveWebsites(websites);

        // Perform initial check
        console.log(`[TitleCheck] Performing initial check for: ${url}`);
        const checkResult = await titleChecker.checkWebsite(newWebsite.url);

        if (checkResult.success) {
            newWebsite.lastTitle = checkResult.title;
            newWebsite.lastStatusCode = checkResult.statusCode;
            newWebsite.lastLoadTime = checkResult.loadTime;
            newWebsite.lastCheck = new Date().toISOString();
            await titleChecker.saveWebsites(websites);
            console.log(`[TitleCheck] Initial check successful: ${checkResult.title}`);

            // Send notification
            console.log(`[TitleCheck] Sending Telegram notification for new website`);
            try {
                const notificationSent = await telegram.sendNewWebsiteAdded(newWebsite, checkResult.title, {
                    loadTime: checkResult.loadTime,
                    statusCode: checkResult.statusCode
                });

                if (notificationSent) {
                    console.log(`[TitleCheck] ✅ Telegram notification sent successfully`);
                } else {
                    console.log(`[TitleCheck] ⚠️ Telegram notification not sent (bot not configured or failed)`);
                }
            } catch (telegramError) {
                console.error(`[TitleCheck] ❌ Error sending Telegram notification:`, telegramError.message);
                // Don't fail the request if notification fails
            }
        } else {
            console.log(`[TitleCheck] Initial check failed for: ${url}`);
        }

        res.json({ success: true, website: newWebsite });
    } catch (error) {
        console.error('[TitleCheck] Error adding website:', error);
        res.status(500).json({ success: false, error: error.message });
    }
});

// Update website
router.put('/:id', async (req, res) => {
    try {
        const { id } = req.params;
        const { url, note } = req.body;

        const websites = await titleChecker.loadWebsites();
        const index = websites.findIndex(w => w.id === id);

        if (index === -1) {
            return res.status(404).json({ success: false, error: 'Website not found' });
        }

        if (url) websites[index].url = url.trim();
        if (note !== undefined) websites[index].note = note.trim();

        await titleChecker.saveWebsites(websites);

        res.json({ success: true, website: websites[index] });
    } catch (error) {
        res.status(500).json({ success: false, error: error.message });
    }
});

// Delete website
router.delete('/:id', async (req, res) => {
    try {
        const { id } = req.params;

        const websites = await titleChecker.loadWebsites();
        const filtered = websites.filter(w => w.id !== id);

        if (filtered.length === websites.length) {
            return res.status(404).json({ success: false, error: 'Website not found' });
        }

        await titleChecker.saveWebsites(filtered);

        res.json({ success: true });
    } catch (error) {
        res.status(500).json({ success: false, error: error.message });
    }
});

// Manual check for specific website
router.post('/:id/check', async (req, res) => {
    try {
        const { id } = req.params;

        const websites = await titleChecker.loadWebsites();
        const website = websites.find(w => w.id === id);

        if (!website) {
            return res.status(404).json({ success: false, error: 'Website not found' });
        }

        // Check if title will change (need screenshot)
        const preCheck = await titleChecker.checkWebsite(website.url, null, false);
        const titleWillChange = website.lastTitle && website.lastTitle !== preCheck.title;

        // Do the actual check with screenshot if needed
        const checkResult = await titleChecker.checkWebsite(website.url, website.id, titleWillChange);

        if (checkResult.success) {
            const titleChanged = website.lastTitle && website.lastTitle !== checkResult.title;
            const statusCodeChanged = website.lastStatusCode && website.lastStatusCode !== checkResult.statusCode;
            const oldTitle = website.lastTitle;
            const oldStatusCode = website.lastStatusCode;

            website.lastTitle = checkResult.title;
            website.lastStatusCode = checkResult.statusCode;
            website.lastLoadTime = checkResult.loadTime;
            website.lastCheck = new Date().toISOString();
            await titleChecker.saveWebsites(websites);

            // Add to history ONLY if title changed (for manual checks)
            if (titleChanged) {
                await titleChecker.addHistoryEntry({
                    id: titleChecker.generateId(),
                    websiteId: website.id,
                    websiteUrl: website.url,
                    websiteNote: website.note,
                    timestamp: website.lastCheck,
                    success: true,
                    title: checkResult.title,
                    loadTime: checkResult.loadTime,
                    statusCode: checkResult.statusCode,
                    titleChanged,
                    oldTitle,
                    statusCodeChanged,
                    oldStatusCode,
                    screenshot: checkResult.screenshot
                });
            }

            // Send notification
            if (titleChanged) {
                await telegram.sendTitleChange(
                    website,
                    oldTitle,
                    checkResult.title,
                    {
                        loadTime: checkResult.loadTime,
                        statusCode: checkResult.statusCode,
                        statusCodeChanged,
                        oldStatusCode
                    },
                    checkResult.screenshot
                );
            } else if (statusCodeChanged) {
                await telegram.sendStatusCodeChange(
                    website,
                    oldStatusCode,
                    checkResult.statusCode,
                    checkResult.screenshot
                );
            } else {
                await telegram.sendTitleUpdate(
                    website,
                    checkResult.title,
                    {
                        loadTime: checkResult.loadTime,
                        statusCode: checkResult.statusCode
                    }
                );
            }
        }

        res.json({ success: true, result: checkResult });
    } catch (error) {
        res.status(500).json({ success: false, error: error.message });
    }
});

module.exports = router;
