const cron = require('node-cron');
const titleChecker = require('./titleChecker');
const telegram = require('./telegram');
const fs = require('fs').promises;
const path = require('path');

class Scheduler {
    constructor() {
        this.settingsPath = path.join(__dirname, '../data/titlecheck/settings.json');
        this.historyPath = path.join(__dirname, '../data/titlecheck/history.json');
        this.task = null;
        this.historyCleanupTask = null;
        this.currentInterval = 15; // Default 15 minutes
    }

    async loadSettings() {
        try {
            const data = await fs.readFile(this.settingsPath, 'utf8');
            return JSON.parse(data);
        } catch (error) {
            // Return default settings
            return {
                telegramToken: '',
                telegramChatId: '',
                checkInterval: 15,
                historyLimit: 1000,
                screenshotEnabled: true
            };
        }
    }

    getCronExpression(minutes) {
        // Convert minutes to cron expression
        if (minutes < 60) {
            return `*/${minutes} * * * *`; // Every X minutes
        } else {
            const hours = Math.floor(minutes / 60);
            return `0 */${hours} * * *`; // Every X hours
        }
    }

    async clearHistory() {
        try {
            console.log('\n🗑️  History cleanup started...');
            await fs.writeFile(this.historyPath, JSON.stringify([], null, 2));
            console.log('✅ History cleared successfully\n');
        } catch (error) {
            console.error('❌ Error clearing history:', error);
        }
    }

    async start() {
        const settings = await this.loadSettings();
        this.currentInterval = settings.checkInterval || 15;

        const cronExpression = this.getCronExpression(this.currentInterval);

        console.log(`[TitleCheck] Starting scheduler with interval: ${this.currentInterval} minutes`);
        console.log(`[TitleCheck] Cron expression: ${cronExpression}`);

        this.task = cron.schedule(cronExpression, async () => {
            console.log('\n=== TitleCheck: Scheduled Check Started ===');
            try {
                const results = await titleChecker.checkAllWebsites();
                await telegram.sendBulkResults(results);
                console.log('=== TitleCheck: Scheduled Check Completed ===\n');
            } catch (error) {
                console.error('[TitleCheck] Error in scheduled check:', error);
            }
        });

        // History cleanup task - runs every 3 days at midnight
        console.log('[TitleCheck] Starting history cleanup task (every 3 days at 00:00)');
        this.historyCleanupTask = cron.schedule('0 0 */3 * *', async () => {
            await this.clearHistory();
        });

        // Run initial check after 10 seconds
        setTimeout(async () => {
            console.log('\n=== TitleCheck: Initial Check Started ===');
            try {
                const results = await titleChecker.checkAllWebsites();
                await telegram.sendBulkResults(results);
                console.log('=== TitleCheck: Initial Check Completed ===\n');
            } catch (error) {
                console.error('[TitleCheck] Error in initial check:', error);
            }
        }, 10000);
    }

    async updateInterval(minutes) {
        console.log(`[TitleCheck] Updating interval from ${this.currentInterval} to ${minutes} minutes`);

        // Stop current task
        if (this.task) {
            this.task.stop();
        }

        this.currentInterval = minutes;

        // Start new task with new interval
        const cronExpression = this.getCronExpression(minutes);
        console.log(`[TitleCheck] New cron expression: ${cronExpression}`);

        this.task = cron.schedule(cronExpression, async () => {
            console.log('\n=== TitleCheck: Scheduled Check Started ===');
            try {
                const results = await titleChecker.checkAllWebsites();
                await telegram.sendBulkResults(results);
                console.log('=== TitleCheck: Scheduled Check Completed ===\n');
            } catch (error) {
                console.error('[TitleCheck] Error in scheduled check:', error);
            }
        });

        return true;
    }

    stop() {
        if (this.task) {
            this.task.stop();
            console.log('[TitleCheck] Scheduler stopped');
        }
        if (this.historyCleanupTask) {
            this.historyCleanupTask.stop();
            console.log('[TitleCheck] History cleanup task stopped');
        }
    }
}

module.exports = new Scheduler();
