const TelegramBot = require('node-telegram-bot-api');
const fs = require('fs').promises;
const path = require('path');

class TelegramService {
    constructor() {
        this.settingsPath = path.join(__dirname, '../data/titlecheck/settings.json');
        this.bot = null;
        this.chatId = null;
        this.init();
    }

    async init() {
        const settings = await this.loadSettings();
        if (settings.telegramToken && settings.telegramChatId) {
            try {
                this.bot = new TelegramBot(settings.telegramToken, { polling: false });
                this.chatId = settings.telegramChatId;
                console.log('[TitleCheck] Telegram bot initialized successfully');
                console.log(`[TitleCheck] Chat ID: ${this.chatId}`);
            } catch (error) {
                console.error('[TitleCheck] Failed to initialize Telegram bot:', error.message);
            }
        } else {
            console.log('[TitleCheck] Telegram not configured (missing token or chatId)');
        }
    }

    async loadSettings() {
        try {
            const data = await fs.readFile(this.settingsPath, 'utf8');
            return JSON.parse(data);
        } catch (error) {
            console.log('[TitleCheck] No settings file found, using defaults');
            // Return default settings
            return {
                telegramToken: '',
                telegramChatId: '',
                checkInterval: 15
            };
        }
    }

    async updateCredentials(token, chatId) {
        try {
            if (token && chatId) {
                this.bot = new TelegramBot(token, { polling: false });
                this.chatId = chatId;
                console.log('[TitleCheck] Telegram credentials updated successfully');
                console.log(`[TitleCheck] New Chat ID: ${chatId}`);
            } else {
                console.log('[TitleCheck] Telegram credentials cleared (empty token or chatId)');
                this.bot = null;
                this.chatId = null;
            }
        } catch (error) {
            console.error('[TitleCheck] Failed to update Telegram credentials:', error.message);
        }
    }

    async sendNotification(message) {
        if (!this.bot || !this.chatId) {
            console.error('[TitleCheck] Telegram bot not initialized');
            return false;
        }

        try {
            await this.bot.sendMessage(this.chatId, message, {
                parse_mode: 'HTML',
                disable_web_page_preview: true
            });
            console.log('[TitleCheck] Telegram notification sent successfully');
            return true;
        } catch (error) {
            console.error('[TitleCheck] Error sending Telegram notification:', error.message);
            return false;
        }
    }

    async sendPhoto(photoPath, caption) {
        if (!this.bot || !this.chatId) {
            console.error('[TitleCheck] Telegram bot not initialized');
            return false;
        }

        try {
            await this.bot.sendPhoto(this.chatId, photoPath, { caption, parse_mode: 'HTML' });
            console.log('[TitleCheck] Telegram photo sent successfully');
            return true;
        } catch (error) {
            console.error('[TitleCheck] Error sending Telegram photo:', error.message);
            return false;
        }
    }

    async sendTitleUpdate(website, title, performance = {}) {
        let message = `
🔍 <b>Title Kontrolü</b>

🌐 <b>Website:</b> ${website.url}
${website.note ? `📝 <b>Not:</b> ${website.note}\n` : ''}📄 <b>Title:</b> ${title}
`;

        // Add performance info if available
        if (performance.loadTime) {
            message += `\n⚡ <b>Performans:</b>\n`;
            message += `   ⏱️ Yüklenme: ${performance.loadTime}ms\n`;
            message += `   📊 Status: ${performance.statusCode}\n`;
        }

        message += `\n⏰ <i>${new Date().toLocaleString('tr-TR')}</i>`;

        return await this.sendNotification(message.trim());
    }

    async sendNewWebsiteAdded(website, title, performance = {}) {
        let message = `
✅ <b>YENİ WEBSITE EKLENDİ</b>

🌐 <b>Website:</b> ${website.url}
${website.note ? `📝 <b>Not:</b> ${website.note}\n` : ''}📄 <b>İlk Title:</b> ${title}
`;

        // Add performance info if available
        if (performance.loadTime) {
            message += `\n⚡ <b>Performans:</b>\n`;
            message += `   ⏱️ Yüklenme: ${performance.loadTime}ms`;

            if (performance.loadTime > 5000) {
                message += ` 🐌 (Yavaş!)`;
            } else if (performance.loadTime < 2000) {
                message += ` ⚡ (Hızlı!)`;
            }

            message += `\n   📊 Status: ${performance.statusCode}\n`;
        }

        const settings = await this.loadSettings();
        const checkInterval = Math.floor(settings.checkInterval || 15);
        message += `\n🎯 <b>İzleme başlatıldı!</b> Her ${checkInterval} dakikada bir kontrol edilecek.\n`;
        message += `\n⏰ <i>${new Date().toLocaleString('tr-TR')}</i>`;

        return await this.sendNotification(message.trim());
    }

    async sendTitleChange(website, oldTitle, newTitle, performance = {}, screenshot = null) {
        let message = `
⚠️ <b>TITLE DEĞİŞTİ!</b>

🌐 <b>Website:</b> ${website.url}
${website.note ? `📝 <b>Not:</b> ${website.note}\n` : ''}📄 <b>Eski Title:</b> ${oldTitle}
✨ <b>Yeni Title:</b> ${newTitle}
`;

        // Add performance info
        if (performance.loadTime) {
            message += `\n⚡ <b>Performans:</b>\n`;
            message += `   ⏱️ Yüklenme: ${performance.loadTime}ms`;

            if (performance.loadTime > 5000) {
                message += ` 🐌 (Yavaş!)`;
            } else if (performance.loadTime < 2000) {
                message += ` ⚡ (Hızlı!)`;
            }

            message += `\n   📊 Status: ${performance.statusCode}`;

            if (performance.statusCodeChanged) {
                message += ` (Önceki: ${performance.oldStatusCode})`;
            }
        }

        message += `\n\n⏰ <i>${new Date().toLocaleString('tr-TR')}</i>`;

        // Send screenshot if available
        if (screenshot) {
            try {
                await this.sendPhoto(screenshot, message.trim());
                console.log('[TitleCheck] Screenshot sent with title change notification');
            } catch (error) {
                // If screenshot fails, send text message
                await this.sendNotification(message.trim());
            }
        } else {
            await this.sendNotification(message.trim());
        }

        return true;
    }

    async sendStatusCodeChange(website, oldStatus, newStatus, screenshot = null) {
        const message = `
🚨 <b>STATUS CODE DEĞİŞTİ!</b>

🌐 <b>Website:</b> ${website.url}
${website.note ? `📝 <b>Not:</b> ${website.note}\n` : ''}📊 <b>Eski Status:</b> ${oldStatus}
📊 <b>Yeni Status:</b> ${newStatus}

${newStatus >= 400 ? '⚠️ <b>UYARI: Site erişilebilir değil!</b>' : ''}

⏰ <i>${new Date().toLocaleString('tr-TR')}</i>
    `.trim();

        if (screenshot) {
            await this.sendPhoto(screenshot, message);
        } else {
            await this.sendNotification(message);
        }

        return true;
    }

    async sendBulkResults(results) {
        // Check if Telegram is configured
        if (!this.bot || !this.chatId) {
            console.log('[TitleCheck] Skipping Telegram notification - bot not configured');
            return false;
        }

        // If no results, notify about that
        if (!results || results.length === 0) {
            console.log('[TitleCheck] No websites to check');
            return await this.sendNotification('📊 <b>Kontrol Özeti</b>\n\n⚠️ Henüz eklenmiş website yok.\n\n⏰ <i>' + new Date().toLocaleString('tr-TR') + '</i>');
        }

        const changedTitles = results.filter(r => r.titleChanged);
        const statusChanges = results.filter(r => r.statusCodeChanged);
        const successfulChecks = results.filter(r => r.success);
        const failedChecks = results.filter(r => !r.success);

        console.log(`[TitleCheck] Sending bulk results: ${results.length} total, ${changedTitles.length} changed, ${failedChecks.length} failed`);

        // Send individual notifications for title changes
        for (const result of changedTitles) {
            await this.sendTitleChange(
                { url: result.websiteUrl, note: result.websiteNote },
                result.oldTitle,
                result.title,
                {
                    loadTime: result.loadTime,
                    statusCode: result.statusCode,
                    statusCodeChanged: result.statusCodeChanged,
                    oldStatusCode: result.oldStatusCode
                },
                result.screenshot
            );
        }

        // Send individual notifications for status code changes (that didn't have title changes)
        for (const result of statusChanges.filter(r => !r.titleChanged)) {
            await this.sendStatusCodeChange(
                { url: result.websiteUrl, note: result.websiteNote },
                result.oldStatusCode,
                result.statusCode,
                result.screenshot
            );
        }

        // Build detailed website list with URLs and titles
        let websiteList = '';
        for (const result of results) {
            const statusIcon = result.success ? '✅' : '❌';
            const changeIcon = result.titleChanged ? ' 🔄' : '';
            const siteName = result.websiteNote || 'Site';
            const url = result.websiteUrl;
            const title = result.success ? result.title : 'Kontrol başarısız';

            websiteList += `\n${statusIcon}${changeIcon} <b>${siteName}</b>\n`;
            websiteList += `🌐 ${url}\n`;
            websiteList += `📄 ${title}\n`;
        }

        // Send detailed summary
        const message = `
📊 <b>Kontrol Özeti</b>

<b>Durum:</b>
✅ Başarılı: ${successfulChecks.length}
${failedChecks.length > 0 ? `❌ Başarısız: ${failedChecks.length}\n` : ''}🔄 Title Değişen: ${changedTitles.length}
📝 Toplam Site: ${results.length}

<b>Kontrol Edilen Siteler:</b>${websiteList}
⏰ <i>${new Date().toLocaleString('tr-TR')}</i>
    `.trim();

        return await this.sendNotification(message);
    }
}

module.exports = new TelegramService();
