const puppeteer = require('puppeteer');
const fs = require('fs').promises;
const path = require('path');

class TitleChecker {
    constructor() {
        this.websitesPath = path.join(__dirname, '../data/titlecheck/websites.json');
        this.historyPath = path.join(__dirname, '../data/titlecheck/history.json');
        this.settingsPath = path.join(__dirname, '../data/titlecheck/settings.json');
        this.screenshotsPath = path.join(__dirname, '../data/titlecheck/screenshots');
        this.browser = null;
    }

    async initBrowser() {
        if (!this.browser) {
            this.browser = await puppeteer.launch({
                headless: 'new',
                args: [
                    '--no-sandbox',
                    '--disable-setuid-sandbox',
                    '--disable-dev-shm-usage',
                    '--disable-accelerated-2d-canvas',
                    '--disable-gpu'
                ]
            });
        }
        return this.browser;
    }

    async checkWebsite(url, websiteId = null, takeScreenshot = false) {
        let page = null;
        const startTime = Date.now();

        try {
            const browser = await this.initBrowser();
            page = await browser.newPage();

            // Set Googlebot user agent
            await page.setUserAgent('Mozilla/5.0 (compatible; Googlebot/2.1; +http://www.google.com/bot.html)');
            await page.setViewport({ width: 1920, height: 1080 });

            // Navigate and track response
            const response = await page.goto(url, {
                waitUntil: 'networkidle2',
                timeout: 30000
            });

            // Get status code
            const statusCode = response.status();

            // Wait for dynamic content (Puppeteer v22 compatible)
            await new Promise(resolve => setTimeout(resolve, 2000));

            // Calculate load time
            const loadTime = Date.now() - startTime;

            // Get title
            const title = await page.title();

            // Take screenshot if requested
            let screenshotPath = null;
            if (takeScreenshot && websiteId) {
                screenshotPath = await this.takeScreenshot(page, websiteId);
            }

            await page.close();

            return {
                success: true,
                title: title.trim() || 'No Title Found',
                statusCode,
                loadTime,
                screenshot: screenshotPath
            };
        } catch (error) {
            console.error(`Error checking ${url}:`, error.message);

            const loadTime = Date.now() - startTime;

            if (page) {
                await page.close().catch(() => { });
            }

            return {
                success: false,
                title: null,
                statusCode: error.message.includes('404') ? 404 : 0,
                loadTime,
                error: error.message
            };
        }
    }

    async takeScreenshot(page, websiteId) {
        try {
            // Create website-specific folder
            const websiteFolder = path.join(this.screenshotsPath, websiteId);
            await fs.mkdir(websiteFolder, { recursive: true });

            // Generate filename
            const timestamp = Date.now();
            const filename = `${timestamp}.png`;
            const screenshotPath = path.join(websiteFolder, filename);

            // Take full page screenshot
            await page.screenshot({
                path: screenshotPath,
                fullPage: true
            });

            // Clean up old screenshots (keep only last 2)
            await this.cleanupOldScreenshots(websiteFolder);

            return screenshotPath;
        } catch (error) {
            console.error('Error taking screenshot:', error.message);
            return null;
        }
    }

    async cleanupOldScreenshots(folder) {
        try {
            const files = await fs.readdir(folder);
            const screenshots = files
                .filter(f => f.endsWith('.png'))
                .map(f => ({
                    name: f,
                    path: path.join(folder, f),
                    time: parseInt(f.replace('.png', ''))
                }))
                .sort((a, b) => b.time - a.time);

            // Keep only the 2 most recent
            for (let i = 2; i < screenshots.length; i++) {
                await fs.unlink(screenshots[i].path).catch(() => { });
            }
        } catch (error) {
            // Folder might not exist yet
        }
    }

    async loadWebsites() {
        try {
            const data = await fs.readFile(this.websitesPath, 'utf8');
            return JSON.parse(data);
        } catch (error) {
            return [];
        }
    }

    async saveWebsites(websites) {
        await fs.writeFile(this.websitesPath, JSON.stringify(websites, null, 2));
    }

    async loadHistory() {
        try {
            const data = await fs.readFile(this.historyPath, 'utf8');
            return JSON.parse(data);
        } catch (error) {
            return [];
        }
    }

    async saveHistory(history) {
        await fs.writeFile(this.historyPath, JSON.stringify(history, null, 2));
    }

    async addHistoryEntry(entry) {
        const history = await this.loadHistory();
        history.unshift(entry); // Add to beginning

        // Load settings to get history limit
        let historyLimit = 1000; // Default
        try {
            const settingsData = await fs.readFile(this.settingsPath, 'utf8');
            const settings = JSON.parse(settingsData);
            historyLimit = settings.historyLimit || 1000;
        } catch (error) {
            // Use default if settings not found
        }

        // Keep only configured number of entries
        if (history.length > historyLimit) {
            history.splice(historyLimit);
        }

        await this.saveHistory(history);
    }

    async checkAllWebsites(saveToHistory = true) {
        const websites = await this.loadWebsites();
        const results = [];

        // Load settings
        let screenshotEnabled = true;
        try {
            const settingsData = await fs.readFile(this.settingsPath, 'utf8');
            const settings = JSON.parse(settingsData);
            screenshotEnabled = settings.screenshotEnabled !== false;
        } catch (error) {
            // Use default
        }

        console.log(`[${new Date().toISOString()}] Checking ${websites.length} websites...`);

        for (const website of websites) {
            // Do a quick check first to see if we need screenshot
            const quickCheck = await this.checkWebsite(website.url, null, false);

            const titleWillChange = website.lastTitle && website.lastTitle !== quickCheck.title;
            const statusWillChange = website.lastStatusCode && website.lastStatusCode !== quickCheck.statusCode;
            const needsScreenshot = screenshotEnabled && (titleWillChange || statusWillChange);

            // Now do the real check with screenshot if needed
            const checkResult = needsScreenshot
                ? await this.checkWebsite(website.url, website.id, true)
                : quickCheck;

            const result = {
                id: this.generateId(),
                websiteId: website.id,
                websiteUrl: website.url,
                websiteNote: website.note,
                timestamp: new Date().toISOString(),
                ...checkResult
            };

            if (checkResult.success) {
                // Check if title changed
                const titleChanged = website.lastTitle && website.lastTitle !== checkResult.title;
                result.titleChanged = titleChanged;
                result.oldTitle = website.lastTitle;

                // Check if status code changed
                const statusCodeChanged = website.lastStatusCode && website.lastStatusCode !== checkResult.statusCode;
                result.statusCodeChanged = statusCodeChanged;
                result.oldStatusCode = website.lastStatusCode;

                // Update website data
                website.lastTitle = checkResult.title;
                website.lastStatusCode = checkResult.statusCode;
                website.lastLoadTime = checkResult.loadTime;
                website.lastCheck = result.timestamp;

                // Add to history only if specified (scheduled checks only)
                if (saveToHistory) {
                    await this.addHistoryEntry(result);
                }
            }

            results.push(result);
        }

        // Save updated websites
        await this.saveWebsites(websites);

        console.log(`[${new Date().toISOString()}] Check completed. ${results.filter(r => r.success).length}/${results.length} successful`);

        return results;
    }

    generateId() {
        return Date.now().toString(36) + Math.random().toString(36).substr(2);
    }

    async cleanup() {
        if (this.browser) {
            await this.browser.close();
            this.browser = null;
            console.log('Browser closed');
        }
    }
}

module.exports = new TitleChecker();
