const axios = require('axios');
const fs = require('fs');

require('dotenv').config();

const GLOBAL_KEY = process.env.CLOUDFLARE_GLOBAL_KEY;
const EMAIL = process.env.CLOUDFLARE_EMAIL;

async function generateToken() {
    console.log('Generating Secure API Token...');

    // Cloudflare Permission IDs (approximate, usually we need to list specific permission group IDs)
    // However, knowing the exact IDs for "Zone:Edit" etc is tricky without querying /user/tokens/permission_groups first.
    // Let's first query permission groups to find the IDs we need.

    try {
        const groupsResp = await axios.get('https://api.cloudflare.com/client/v4/user/tokens/permission_groups', {
            headers: {
                'X-Auth-Email': EMAIL,
                'X-Auth-Key': GLOBAL_KEY,
                'Content-Type': 'application/json'
            }
        });

        // Zone Level Permissions
        const zoneGroups = [
            { id: 'e6d2666161e84845a636613608cee8d5', name: 'Zone Write' },
            { id: '4755a26eedb94da69e1066d98aa820be', name: 'DNS Write' },
            { id: 'c03055bc037c4ea9afb9a9f104b7b721', name: 'SSL and Certificates Write' },
            { id: 'ed07f6c337da4195b4e72a1fb2c6bcae', name: 'Page Rules Write' },
            { id: '3030687196b94b638145a3953da2b699', name: 'Zone Settings Write' },
            { id: 'fb6778dc191143babbfaa57993f1d275', name: 'Zone WAF Write' },
            { id: '9ff81cbbe65c400b97d92c3c1033cab6', name: 'Cache Settings Write' },
            { id: '3b94c49258ec4573b06d51d99b6416c0', name: 'Bot Management Write' }
        ];

        // Account Level Permissions
        const accountGroups = [
            { id: 'c1fde68c7bcc44588cbb6ddbc16d6480', name: 'Account Settings Read' },
            { id: 'eb56a6953c034b9d97dd838155666f06', name: 'Account API Tokens Read' } // Optional but good for verification
        ];

        console.log('Sending Payload with Split Policies...');

        // Create Token
        const createResp = await axios.post('https://api.cloudflare.com/client/v4/user/tokens', {
            name: 'AutoManager_Generated_Token_PRO_V3',
            policies: [
                {
                    effect: 'allow',
                    resources: {
                        'com.cloudflare.api.account.zone.*': '*' // All zones
                    },
                    permission_groups: zoneGroups.map(p => ({ id: p.id }))
                },
                {
                    effect: 'allow',
                    resources: {
                        'com.cloudflare.api.account.*': '*' // All accounts
                    },
                    permission_groups: accountGroups.map(p => ({ id: p.id }))
                }
            ]
        }, {
            headers: {
                'X-Auth-Email': EMAIL,
                'X-Auth-Key': GLOBAL_KEY,
                'Content-Type': 'application/json'
            }
        });

        if (createResp.data.success) {
            console.log('\nSUCCESS! YOUR NEW TOKEN IS:');
            console.log('==================================================');
            console.log(createResp.data.result.value);
            console.log('==================================================');
        } else {
            console.error('Failed to create token:', createResp.data.errors);
        }

    } catch (error) {
        console.error('Error:', error.response?.data || error.message);

        // If complex to find IDs, we might fallback to a simpler request or ask user to create it.
        // But let's try to list groups first.
        if (error.response?.data?.errors) {
            console.log('Cloudflare Errors:', JSON.stringify(error.response.data.errors, null, 2));
        }
    }
}

// Adjusted: We can't map permissions perfectly by name string matching if names vary. 
// A safer way is using hardcoded IDs if known, or just requesting "Zone:Edit" broad scope if possible?
// No, API requires specific Permission Group IDs.
// Correct approach: Fetch all groups, substring match "Zone", "DNS" etc, and pick "Edit".

generateToken();
