/**
 * WHM API Helper
 * cPanel & WHM v132+ JSON API
 * 
 * Base URL: https://hostname:2087/json-api/
 * Auth: Authorization: whm root:API_TOKEN
 */

const axios = require('axios');
const https = require('https');

const httpsAgent = new https.Agent({ rejectUnauthorized: false });

/**
 * Make a WHM API request
 */
async function whmApi(endpoint, params = {}, config) {
    let { hostname, port = 2087, apiToken, username = 'root' } = config;

    // Clean hostname
    hostname = hostname.replace(/^https?:\/\//, '').replace(/:\d+$/, '').replace(/\/$/, '');
    port = parseInt(port) || 2087;

    const queryParams = new URLSearchParams({
        'api.version': '1',
        ...params
    });

    const url = `https://${hostname}:${port}/json-api/${endpoint}?${queryParams.toString()}`;

    console.log(`[WHM API] ${endpoint} - URL: ${url.substring(0, 100)}...`);

    try {
        const response = await axios.get(url, {
            headers: {
                'Authorization': `whm ${username}:${apiToken}`
            },
            httpsAgent
        });

        console.log(`[WHM API] Response status:`, response.data?.metadata?.result || 'ok');
        return response.data;
    } catch (error) {
        console.error(`[WHM API] Error [${endpoint}]:`, error.response?.data || error.message);
        throw error.response?.data || error;
    }
}

/**
 * Test API connection
 */
async function testConnection(config) {
    try {
        const result = await whmApi('version', {}, config);
        if (result && result.version) {
            return { success: true, version: result.version };
        }
        return { success: true, result };
    } catch (error) {
        return { success: false, error: error.message || 'Connection failed' };
    }
}

/**
 * List available hosting packages
 */
async function listPackages(config) {
    const result = await whmApi('listpkgs', {}, config);
    return result;
}

/**
 * Create a new cPanel account
 */
async function createAccount(accountData, config) {
    const { domain, username, password, plan, contactemail } = accountData;

    const finalUsername = username || domain.replace(/\./g, '').substring(0, 8).toLowerCase();
    const finalPassword = password || generatePassword(16);

    const params = {
        domain,
        username: finalUsername,
        password: finalPassword,
        plan: plan || 'default',
        contactemail: contactemail || '',
        dkim: 1,
        spf: 1
    };

    const result = await whmApi('createacct', params, config);

    if (result.metadata?.result === 1 || result.result?.[0]?.status === 1) {
        return {
            success: true,
            username: finalUsername,
            password: finalPassword,
            domain,
            result
        };
    } else {
        const errorMsg = result.metadata?.reason || result.result?.[0]?.statusmsg || 'Account creation failed';
        throw new Error(errorMsg);
    }
}

/**
 * Add subdomain to an account
 */
async function addSubdomain(subdomain, rootdomain, cpanelUser, config) {
    let { hostname, port = 2087, apiToken, username = 'root' } = config;

    hostname = hostname.replace(/^https?:\/\//, '').replace(/:\d+$/, '').replace(/\/$/, '');
    port = parseInt(port) || 2087;

    const params = new URLSearchParams({
        'cpanel_jsonapi_user': cpanelUser,
        'cpanel_jsonapi_module': 'SubDomain',
        'cpanel_jsonapi_func': 'addsubdomain',
        'cpanel_jsonapi_apiversion': '3',
        'domain': subdomain,
        'rootdomain': rootdomain
    });

    const url = `https://${hostname}:${port}/json-api/cpanel?${params.toString()}`;

    console.log(`[WHM API] Adding subdomain: ${subdomain}.${rootdomain}`);

    try {
        const response = await axios.get(url, {
            headers: {
                'Authorization': `whm ${username}:${apiToken}`
            },
            httpsAgent
        });

        const data = response.data;
        if (data.result?.status === 1 || data.cpanelresult?.data?.[0]?.result === 1) {
            return { success: true, subdomain: `${subdomain}.${rootdomain}` };
        } else {
            const errorMsg = data.result?.errors?.[0] || data.cpanelresult?.data?.[0]?.reason || 'Subdomain creation failed';
            throw new Error(errorMsg);
        }
    } catch (error) {
        console.error(`[WHM API] Subdomain error:`, error.response?.data || error.message);
        throw error.response?.data || error;
    }
}

function generatePassword(length = 16) {
    const chars = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789!@#$%^&*';
    let password = '';
    for (let i = 0; i < length; i++) {
        password += chars.charAt(Math.floor(Math.random() * chars.length));
    }
    return password;
}

function generateUsername(domain) {
    const name = domain.split('.')[0].replace(/[^a-zA-Z0-9]/g, '').substring(0, 8).toLowerCase();
    if (name.length < 4) {
        return name + Math.random().toString(36).substring(2, 6);
    }
    return name;
}

/**
 * Execute cPanel UAPI function as Root via WHM wrapper
 */
async function uapi(cpanelUser, module, functionName, args = {}, config) {
    let { hostname, port = 2087, apiToken, username = 'root' } = config;
    hostname = hostname.replace(/^https?:\/\//, '').replace(/:\d+$/, '').replace(/\/$/, '');

    // Construct URL
    const url = `https://${hostname}:${port}/json-api/cpanel`;

    const params = new URLSearchParams({
        'cpanel_jsonapi_user': cpanelUser,
        'cpanel_jsonapi_apiversion': '3',
        'cpanel_jsonapi_module': module,
        'cpanel_jsonapi_func': functionName,
        ...args
    });

    console.log(`[WHM API] UAPI Call: ${module}::${functionName} for ${cpanelUser}`);

    try {
        const response = await axios.get(`${url}?${params.toString()}`, {
            headers: { 'Authorization': `whm ${username}:${apiToken}` },
            httpsAgent
        });

        return response.data;
    } catch (error) {
        console.error(`[WHM API] UAPI Error:`, error.message);
        throw error;
    }
}

/**
 * Upload file to cPanel account
 * Supports:
 * 1. User Auth via Port 2083 (Preferred for files, requires password)
 * 2. Root Wrapper via Port 2087 (Fallback, inconsistent for files)
 */
async function uploadFile(targetUsername, fileBuffer, filename, destDir = '/public_html', config) {
    let { hostname, port = 2087, apiToken, username = 'root', cpanelPassword } = config;
    hostname = hostname.replace(/^https?:\/\//, '').replace(/:\d+$/, '').replace(/\/$/, '');

    const FormData = require('form-data');

    // STRATEGY 1: Direct User Auth (Port 2083) - If password available
    if (cpanelPassword) {
        console.log(`[WHM API] Uploading via User Auth (2083) for ${targetUsername}...`);
        const url = `https://${hostname}:2083/execute/Fileman/upload_files`;

        const userAuthForm = new FormData();
        userAuthForm.append('dir', destDir);
        userAuthForm.append('file-1', fileBuffer, {
            filename: filename,
            contentType: 'application/zip'
        });

        try {
            const response = await axios.post(url, userAuthForm, {
                auth: { username: targetUsername, password: cpanelPassword },
                headers: { ...userAuthForm.getHeaders() },
                httpsAgent
            });

            const result = response.data;
            if (result.status === 1 || (result.data?.uploads?.length > 0)) {
                console.log(`[WHM API] Upload success:`, result.data?.uploads?.[0]?.file || 'OK');
                return { success: true, ...result };
            } else {
                throw new Error(result.errors?.[0] || "Upload failed");
            }
        } catch (error) {
            console.error(`[WHM API] User Auth Upload error:`, error.message);
            // Don't fallback automatically to Wrapper for files as it is broken. Throw.
            throw error;
        }
    }

    // STRATEGY 2: Root Wrapper (Port 2087)

    console.log(`[WHM API] Uploading via Root Wrapper (2087) for ${targetUsername}...`);
    // Wrapper parameters
    const wrapperForm = new FormData();
    wrapperForm.append('cpanel_jsonapi_user', targetUsername);
    wrapperForm.append('cpanel_jsonapi_apiversion', '3');
    wrapperForm.append('cpanel_jsonapi_module', 'Fileman');
    wrapperForm.append('cpanel_jsonapi_func', 'upload_files');
    wrapperForm.append('dir', destDir);
    wrapperForm.append('file-1', fileBuffer, {
        filename: filename,
        contentType: 'application/zip'
    });

    const url = `https://${hostname}:${port}/json-api/cpanel`;

    try {
        const response = await axios.post(url, wrapperForm, {
            headers: {
                ...wrapperForm.getHeaders(),
                'Authorization': `whm ${username}:${apiToken}`
            },
            httpsAgent
        });

        const result = response.data?.result;
        if (!result) throw new Error("Invalid API response");

        if (result.status === 1 || (result.data?.uploads?.length > 0)) {
            console.log(`[WHM API] Upload success:`, result.data?.uploads?.[0]?.file || 'OK');
            return { success: true, ...result };
        } else {
            throw new Error(result.errors?.[0] || "Upload failed");
        }
    } catch (error) {
        console.error(`[WHM API] Upload error:`, error.message);
        throw error;
    }
}

/**
 * Extract ZIP file on cPanel server using API 2 (Fileman::fileop)
 */
async function extractZip(targetUsername, zipPath, destDir, config) {
    console.log(`[WHM API] Extracting ${zipPath} (API 2)...`);

    let { hostname, port = 2087, apiToken, username = 'root' } = config;
    hostname = hostname.replace(/^https?:\/\//, '').replace(/:\d+$/, '').replace(/\/$/, '');

    const url = `https://${hostname}:${port}/json-api/cpanel`;
    const params = new URLSearchParams({
        'cpanel_jsonapi_user': targetUsername,
        'cpanel_jsonapi_apiversion': '2',
        'cpanel_jsonapi_module': 'Fileman',
        'cpanel_jsonapi_func': 'fileop',
        'op': 'extract',
        'sourcefiles': zipPath,
        'destfiles': destDir,
        'doubledecode': '0'
    });

    try {
        const response = await axios.get(`${url}?${params.toString()}`, {
            headers: { 'Authorization': `whm ${username}:${apiToken}` },
            httpsAgent
        });

        const data = response.data;
        // Check API 2 result
        if (data.cpanelresult?.data?.[0]?.result === 1) {
            console.log(`[WHM API] Extract success`);
            return { success: true };
        } else {
            const error = data.cpanelresult?.data?.[0]?.reason ||
                data.cpanelresult?.error ||
                "Extraction failed without specific reason";
            throw new Error(error);
        }
    } catch (error) {
        console.error(`[WHM API] Extract error:`, error.message);
        throw error;
    }
}

/**
 * List files in directory using UAPI Wrapper
 */
async function listFiles(targetUsername, dir, config) {
    console.log(`[WHM API] Listing ${dir} (Wrapper)...`);

    const res = await uapi(targetUsername, 'Fileman', 'list_files', {
        'dir': dir
    }, config);

    if (res.result?.status === 1) {
        return res.result; // contains .data array
    } else {
        throw new Error(res.result?.errors?.[0] || 'List failed');
    }
}

module.exports = {
    whmApi,
    testConnection,
    listPackages,
    createAccount,
    addSubdomain,
    uploadFile,
    extractZip,
    listFiles,
    generatePassword,
    generateUsername
};
